import logging
logging.basicConfig(level=logging.DEBUG, format="[%(asctime)s] %(levelname)s %(name)s %(message)s")
logger = logging.getLogger(__name__)

import statface_client
import os
import pandas as pd
import numpy as np
from lib.util import date_to_timestamp
from lib.decorators import cached, retry
import datetime

from lib.constants import *

from collections import namedtuple




class StatCollector(object):

    def __init__(self, path, prod=True, capitalize=True):
        self.stat_token = os.getenv('STAT_TOKEN')
        self.client = statface_client.StatfaceClient(oauth_token=self.stat_token, host=statface_client.STATFACE_PRODUCTION)
        cap = str.capitalize if capitalize else str
        path = '/'.join(map(cap, path.split('/')))
        self.report = self.client.get_report(path)

    @retry(3, 1, 1)
    def download_data(self, interval, agg='d', **kwargs):
        '''
        :param interval: int or pair of min/max dates
        :return:
        '''
        days, date_min, date_max = None, None, None

        try:
            dt = list(interval)
            date_min, date_max = map(
                lambda x: datetime.datetime.fromtimestamp(date_to_timestamp(x)).strftime('%Y-%m-%d'),
                dt
            )
        except TypeError:
            days = interval

        config = self.report.download_config()
        if not (date_min and date_max):
            data = self.report.download_data(agg, _period_distance=days, **kwargs)
        else:
            data = self.report.download_data(agg, date_min=date_min, date_max=date_max, **kwargs)
        return data

    @cached()
    def get_config(self):
        return self.report.download_config()

    def _filter_data(self, data, fields=()):
        cnf = self.get_config()
        if not fields:
            fields = [k for conf in cnf['fields'] for k in conf.keys()]
        return [{k:v for k,v in d.items() if k in fields} for d in data]


class DiskDAUStatCollector(StatCollector):

    def __init__(self, path='Disk/Special/Audience_v2'):
        super(DiskDAUStatCollector, self).__init__(path, True)

    def download_data(self, interval, agg='d', **kwargs):
        return super(DiskDAUStatCollector, self).download_data(interval, agg=agg, platform='_total_', is_payment='_total_',
                                                               action__lvl='1', puid_type='_total_', max_distance='1',
                                                               _incl_fields='users', action='\t_total_\tactive\t',
                                                               _apply_dictionary=0, **kwargs)

    def collect_dau(self, interval, keys=('fielddate', 'users')):
        data = self.download_data(interval)
        return [{k:v for k,v in d.items() if k in ('fielddate', 'users')} for d in data]

    def collect_wau(self, interval, keys=('fielddate', 'users')):
        data = self.download_data(interval, agg='w')
        return [{k:v for k,v in d.items() if k in ('fielddate', 'users')} for d in data]


class DiskWauForecast022020StatCollector(StatCollector):

    def __init__(self, path='Disk/Audience/WAUForecast'):
        super(DiskWauForecast022020StatCollector, self).__init__(path, True, capitalize=False)

    def download_data(self, interval, to_df=True, date_field='total_disk_072021', agg='w', **kwargs):
        '''scale=w&_incl_fields=total_disk_022020&date_min=2018-11-05+00%3A00%3A00&date_max=2021-05-09+23%3A59%3A59'''
        data = super(DiskWauForecast022020StatCollector, self).download_data(interval, agg='w', _incl_fields=date_field, _apply_dictionary=0, **kwargs)
        data = [{k:v for k, v in d.items() if k in (fd, date_field)} for d in data]
        if not to_df:
            return data
        df = pd.DataFrame(data)
        df[fd] = pd.to_datetime(df[fd])
        df = df.sort_values(fd).set_index(fd)
        df.rename(columns={date_field: 'wau'}, inplace=True)
        return df.reindex(pd.date_range(df.index[0], df.index[-1]), fill_value=np.nan).interpolate()


class DiskResourcesStatCollector(StatCollector):

    def __init__(self, path='Disk/Admin/Resources', *args, **kwargs):
        super(DiskResourcesStatCollector, self).__init__(path, *args, **kwargs)

    def download_data(self, interval, agg='d', filt=True, **kwargs):
        data = super(DiskResourcesStatCollector, self).download_data(interval, agg, **kwargs)
        if filt:
            data = self._filter_data(data)
        return data

    def get_data(self, prj, interval):
        pass

    def get_qloud_data(self, prj, interval):
        pass

