# -*- coding: utf-8 -*-

import re
import requests
import time

from collections import namedtuple

from yasmapi import GolovanRequest
GolovanRequest.GOLOVAN_PATH = 'hist/series'
from yasmapi.hist import RetryLimitExceeded

from lib.collectors.conductor import conductor
from lib.decorators import cached
from lib.util import date_to_timestamp, reverse_dict, get_url, to_float, get_nth_percentile, HOUR, DAY, FIVE_M

import logging
logging.basicConfig(level=logging.DEBUG, format="[%(asctime)s] %(levelname)s %(name)s %(message)s")
logger = logging.getLogger(__name__)


class Signal:
    allowed_params = {'itype', 'hosts', 'prj', 'ctype', 'geo', 'tier', 'graphs', 'charts', 'signals', 'stage'}
    uniq_params = {'graphs', 'charts', 'signals'}
    '''
        * 5 — Пятисекундные данные
        * 300 — Пятиминутные данные, расчитываются раз в 25 минут
        * 3600 — Часовые данные, расчитываются раз в час
        * 10800 — Трёхчасовые данные, расчитываются раз в три часа
        * 21600 — Шестичасовые данные, расчитываются раз в шесть часов
        * 43200 — Двенадцатичасовые данные, расчитываются раз в двенадцать часов
        * 86400 — Суточные данные, расчитываются раз в сутки
    '''

    @classmethod
    def generate(cls, **kwargs):
        for k in kwargs:
            if k not in cls.allowed_params:
                raise AttributeError('{} is not in allowed parameters: {}'.format(k, cls.allowed_params))

        sig_key = set(kwargs.keys()).intersection(cls.uniq_params)
        if len(sig_key) != 1:

            raise AttributeError('Set only one of allowed signal parameters: {}'.format(cls.uniq_params))

        sig = kwargs.pop(sig_key.pop())

        sig_string = '{}:{}'.format(';'.join('{}={}'.format(k, v) for k, v in kwargs.items()), sig)
        return sig_string

class YasmSig:
    def __init__(self, value, agg_method='max', agg_time=FIVE_M):
        self.value = value
        self.agg_method = agg_method
        self.agg_time = agg_time

    def __str__(self):
        return 'Sig(Value: %s, agg_method: %s, agg_time: %s)' % (self.value, self.agg_method, self.agg_time)

    def __repr__(self):
        return str({'value': self.value, 'agg_method': self.agg_method, 'agg_time': self.agg_time})


class YasmCollector(object):
    allowed_times = {5, 300, 3600, 10800, 21600, 43200, 86400}
    HOUR = 3600
    DAY = 86400
    SHARPEI_GROUPS = ('disk_sharpei',)
    DB_GROUPS = ('disk_pg', 'disk_test_qdb', 'disk_sharpei_datasync_qa')
    QLOUD_GROUPS = ('disk_qloud',)
    DEPLOY_GROUPS = ('disk_deploy',)

    SIGNALS = {
        'cpu_total': YasmSig('sum(havg(cpu-usage_cores_hgram), havg(cpu-idle_cores_hgram))', 'aver'),
        'cpu_usage': YasmSig('quant(cpu-usage_cores_hgram, max)', '98'),
        'disk_total': YasmSig('sum(disk-avail_space_gb_/_tnnv, disk-usage_space_gb_/_tmmv)', 'aver'),
        'disk_usage': YasmSig('disk-usage_space_gb_/_tmmv', 'max'),
        'mem_total': YasmSig('div(sum(havg(mem-cache_thhh), havg(mem-used_thhh), havg(mem-mem_free_thhh)), 1024)', 'aver'),
        'mem_usage': YasmSig('div(sum(havg(mem-cache_thhh), havg(mem-used_thhh)), 1024)', 'max')
    }

    def __init__(self, *args, **kwargs):
        self.signals = {}
        self.init_signals(self.SIGNALS)


    def init_signals(self, rewrite_signals=None):
        if rewrite_signals:
            self.signals.update(rewrite_signals)

    def collect_signal(self, host, period, start_timestamp, end_timestamp=None, timestamp_delta=DAY, signal=None, **kwargs):
        end_timestamp = end_timestamp or start_timestamp + timestamp_delta - 1 # exclude end timestamp point
        signal = signal or Signal.generate(**kwargs)
        if period not in self.allowed_times:
            raise ValueError("Interval %s is not one of %s" % (period, self.allowed_times))
        data = list(GolovanRequest(host, period, start_timestamp, end_timestamp, [signal], max_retry=3, explicit_fail=True))
        data = [d[1][signal] for d in data]
        return data

    def _get_sig(self, fqdn, signal, ts, period=FIVE_M):
        return self.collect_signal(fqdn, period, start_timestamp=ts, itype=self.ITYPE, prj=self.PRJ, signals=signal)

    def get_sig(self, fqdn, signal, date, retry_sig_interval=HOUR):
        ts = date_to_timestamp(date, round_to_day=True)
        res = self._get_sig(fqdn, signal.value, ts, signal.agg_time)
        # Try to recollect with hour step if no data collected
        if all(r is None for r in res) and retry_sig_interval:
            logger.warning('No data collected: (%s, %s, %s)' % (fqdn, signal.value, date))
            logger.info('Recollecting data with period %s' % (DAY))
            res = self._get_sig(fqdn, signal.value, ts, period=retry_sig_interval)

        return self._agg_data(res, signal.agg_method)


    def _agg_data(self, data, agg_method):
        '''
        :param data:
        :param agg_method: aver/max/quantile
        :return:
        '''
        if not any(data):
            return None
        tdata = [to_float(x) for x in data]
        if agg_method == 'aver':
            return sum(tdata)/(len(tdata) or 1)
        if agg_method == 'max':
            return max(tdata)
        return get_nth_percentile(data, int(to_float(agg_method)))


    @cached(days=999)
    def collect_stats(self, fqdn, date):
        date = date_to_timestamp(date, round_to_day=True)
        results = {
            'date': date,
            'fqdn': fqdn,
        }
        for name, sig in self.signals.items():
            try:
                results[name] = self.get_sig(fqdn, sig, date)
            except RetryLimitExceeded:
                logger.warning('Signal "%s" is not collected from %s' % (sig, fqdn))
                return results
        return results


class YasmQloudCollector(YasmCollector):

    # Overall in QloudCollector fqdn present qloud's project
    QLOUD_REQ = 'http://yasm.yandex-team.ru/metainfo/tags/prj?itype=qloud'
    BASE_PATTERN = re.compile('^(disk\.)')
    HOST = "QLOUD"
    ITYPE = 'qloud'

    SIGNALS = {
        'cpu_usage': YasmSig('mul(quant(portoinst-cpu_usage_cores_hgram, 95), counter-instance_tmmv)'),
        'cpu_total': YasmSig('portoinst-cpu_limit_cores_tmmv'),
        'disk_usage': YasmSig('mul(conv(portoinst-volume_root_usage_txxx, Gi), counter-instance_tmmv)'),
        'disk_total': YasmSig('mul(conv(portoinst-volume_root_quota_txxx, Gi), counter-instance_tmmv)'),
        'mem_usage': YasmSig('portoinst-memory_usage_gb_tmmv'),
        'mem_total': YasmSig('portoinst-memory_limit_gb_tmmv'),
    }

    def __init__(self, *args, **kwargs):
        super(YasmQloudCollector, self).__init__(*args, **kwargs)
        self.init_signals(self.SIGNALS)

    def _get_projects(self, re_pattern=re.compile('.*')):
        r = get_url(self.QLOUD_REQ)
        raw_projects = r['response']['result']
        return tuple(filter(lambda x: re_pattern.search(x), raw_projects))

    @cached(days=999)
    def get_all_hosts(self):
        projects = self._get_projects(self.BASE_PATTERN)
        return {'qloud': projects}

    def _get_sig(self, fqdn, signal, ts, period=FIVE_M):
        return self.collect_signal(self.HOST, period,  start_timestamp=ts, prj=fqdn, itype=self.ITYPE, signals=signal)


class YasmDeployCollector(YasmCollector):

    DEPLOY_STAGES = 'http://yasm.yandex-team.ru/metainfo/tags/stage?itype=deploy'
    BASE_PATTERN = re.compile(r'^(disk|telemost|jicofo)')
    HOST = 'ASEARCH'
    ITYPE = 'deploy'

    SIGNALS = {
        'cpu_usage': YasmSig('mul(quant(portoinst-cpu_usage_cores_hgram, 95), counter-instance_tmmv)'),
        'cpu_total': YasmSig('portoinst-cpu_limit_cores_tmmv'),
        'disk_usage': YasmSig('portoinst-anon_usage_gb_tmmv'),
        'disk_total': YasmSig('portoinst-anon_limit_tmmv'),
        'mem_usage': YasmSig('portoinst-memory_usage_gb_tmmv'),
        'mem_total': YasmSig('portoinst-memory_limit_gb_tmmv'),
    }

    def __init__(self, *args, **kwargs):
        super(YasmDeployCollector, self).__init__(*args, **kwargs)
        # self.projects = self.get_all_hosts()
        self.init_signals(self.SIGNALS)

    def _get_projects(self, re_pattern=re.compile('.*')):
        r = get_url(self.DEPLOY_STAGES)
        raw_projects = r['response']['result']
        return tuple(filter(lambda x: re_pattern.search(x), raw_projects))

    @cached(days=999)
    def get_all_hosts(self):
        projects = self._get_projects(self.BASE_PATTERN)
        return {'deploy': projects}

    def _get_sig(self, stage, signal, ts, period=FIVE_M):
        return self.collect_signal(self.HOST, period,  start_timestamp=ts, stage=stage, itype=self.ITYPE, signals=signal)


class YasmHardwareCollector(YasmCollector):
    ITYPE='disk'
    PRJ='disk'

    def __init__(self, *args, **kwargs):
        super(YasmHardwareCollector, self).__init__(*args, **kwargs)
        self.RESTRICTED_GROUPS = self.QLOUD_GROUPS + self.DB_GROUPS + self.SHARPEI_GROUPS + self.DEPLOY_GROUPS

    def get_all_hosts(self):
        hosts = conductor.get_groups_with_hosts_by_prj('disk', recursive=True)
        groups_per_hosts = reverse_dict(hosts)
        hw_hosts = {}
        for grp, hsts in hosts.items():
            add_group = True
            for db_group in self.RESTRICTED_GROUPS:
                if any(db_group in groups_per_hosts[hst] for hst in hsts):
                    add_group = False
            if add_group:
                hw_hosts.update({grp: hsts})
        return hw_hosts


class YasmPGCollector(YasmHardwareCollector):
    ITYPE='mailpostgresql'
    PRJ=''

    SIGNALS = {
        'cpu_total': YasmSig('sum(havg(cpu-usage_cores_hgram), havg(cpu-idle_cores_hgram))', 'aver', HOUR),
        'cpu_usage': YasmSig('quant(cpu-usage_cores_hgram, 95)', 'max', HOUR),
        'disk_usage': YasmSig('''
                                conv(
                                    or(or(
                                        push-disk-used_bytes_/var/lib/postgresql/9.6/data_tmmv,
                                        push-disk-used_bytes_/var/lib/postgresql/10/data_tmmv
                                    ), push-disk-used_bytes_/_tmmv),
                                    Gi
                                )'''.replace(' ', '').replace('\n', ''),
                              'max', HOUR),
        'disk_total': YasmSig('''
                                conv(
                                    or(or(
                                        push-disk-total_bytes_/var/lib/postgresql/9.6/data_tmmv,
                                        push-disk-total_bytes_/var/lib/postgresql/10/data_tmmv
                                    ), push-disk-total_bytes_/_tmmv),
                                    Gi
                                )'''.replace(' ', '').replace('\n', ''),
                              'max', HOUR),
        'mem_total': YasmSig('div(sum(havg(mem-cache_thhh), havg(mem-used_thhh), havg(mem-mem_free_thhh)), 1024)', 'aver'),
        'mem_usage': YasmSig('div(sum(havg(mem-cache_thhh), havg(mem-used_thhh)), 1024)', 'max')
    }

    def __init__(self, *args, **kwargs):
        super(YasmPGCollector, self).__init__(*args, **kwargs)
        self.init_signals(self.SIGNALS)

    def _get_sig(self, fqdn, signal, ts, period=HOUR):
        return self.collect_signal(fqdn, period, start_timestamp=ts, itype=self.ITYPE, signals=signal)

    @cached(days=999)
    def get_all_hosts(self):
        hosts = conductor.get_groups_with_hosts_by_prj('disk',
                                                       lambda x: 'qloud' not in x and 'cloud' not in x,
                                                       recursive=True)
        groups_per_hosts = reverse_dict(hosts)
        db_hosts = {}
        # Filter group+host only if host belongs to specific group
        for grp, hsts in hosts.items():
            for db_group in self.DB_GROUPS:
                if any(db_group in groups_per_hosts[hst] for hst in hsts):
                    db_hosts.update({grp:hsts})
                    break
        return db_hosts


class YasmSharpeiCollector(YasmPGCollector):
    ITYPE = 'mailsharpei'
    @cached(days=999)
    def get_all_hosts(self):
        hosts = conductor.get_groups_with_hosts_by_prj('disk', recursive=True)
        groups_per_hosts = reverse_dict(hosts)
        db_hosts = {}
        # Filter group+host only if host belongs to specific group
        for grp, hsts in hosts.items():
            for db_group in self.SHARPEI_GROUPS:
                if any(db_group in groups_per_hosts[hst] for hst in hsts):
                    db_hosts.update({grp:hsts})
                    break
        return db_hosts


class YasmMDBCollector(YasmCollector):
    ITYPE='mailpostgresql'
    PSEUDO_CGROUP='mdb'

    SIGNALS = {
        'cpu_total': YasmSig('sum(havg(cpu-usage_cores_hgram), havg(cpu-idle_cores_hgram))', 'aver'),
        'cpu_usage': YasmSig('quant(cpu-usage_cores_hgram, max)', '98'),
        'disk_total': YasmSig('conv(push-disk-total_bytes_pgdata_tmmx, Gi)', 'aver'),
        'disk_usage': YasmSig('conv(push-disk-used_bytes_pgdata_tmmx, Gi)', 'max'),
        'mem_total': YasmSig('div(sum(havg(mem-cache_thhh), havg(mem-used_thhh), havg(mem-mem_free_thhh)), 1024)', 'aver'),
        'mem_usage': YasmSig('div(sum(havg(mem-cache_thhh), havg(mem-used_thhh)), 1024)', 'max')
    }

    def __init__(self, *args, **kwargs):
        super(YasmMDBCollector, self).__init__(*args, **kwargs)
        self.init_signals(self.SIGNALS)

    def _get_sig(self, host, signal, ts, period=FIVE_M):
        return self.collect_signal('CON', period, start_timestamp=ts, itype=self.ITYPE, ctype=host, signals=signal)

    def get_all_hosts(self):
        #TODO: разобраться с IAM токенами облака, реализовать нормальную динамическую сборку хостов
        mdb_host = namedtuple('mdb_host', ('name', 'id'))

        hosts = {self.PSEUDO_CGROUP: []}
        for h in [
            ("datasync-01", "43e9f9c8-cb2a-4491-ab83-ec6fc5639049"),
            ("datasync-02", "c01c3063-8a90-4636-8cae-eb0e06b1c76c"),
            ("datasync-03", "d95ed886-5a3a-4e9f-ae79-e6002177aba4"),
            ("datasync-04", "4e352527-2517-44d9-bb1e-5d310ee2e86c"),
            ("datasync-05", "96e38458-e2c5-4b53-a3ec-d39bb26722f6"),
            ("datasync-06", "9226f3d1-29e7-441f-b58d-9e42b585dc85"),
            ("datasync-07", "55793098-692c-40b6-bd0b-0c28b7bbdc90"),
            ("datasync-08", "13a18a2d-981e-4c68-9927-3735f2eb84ea"),
            ("datasync-09", "5f249a80-25f5-48a7-92c3-a6ba3c0b84f0"),
            ("datasync-10", "mdbo56iaamqov23j5ktq"),
            ("datasync-11", "mdbre9dafdii1hofjgsr"),
            ("datasync-12", "mdbqqc3fkek0p5qedmdh"),
            ("datasync-13", "mdbdtge9nr5fcsuqkti8"),
            ("datasync-14", "mdbjbdmdj2opqnm9qe80"),
            ("datasync-15", "mdbvui26bht9hiu7r6u7"),
            ("datasync-16", "mdb1puok1gd8pg4rdr4o"),
            ("datasync-17", "mdbk7s3fv9r636cd7mbr"),
            ("datasync-18", "mdbd22lalpovndi7ndka"),
            ("datasync-19", "mdblfvks4tmero7rnmuq"),
            ("datasync-20", "mdb76eipcmp2uq8dni20"),
            ("datasync-21", "mdbe9618j01fp2o9o2i7"),
            ("datasync-22", "mdb7tjrif5tcjoi00f0r"),
            ("datasync-23", "mdbphpuho2v360j2bruf"),
            ("datasync-24", "mdboaaqbqcurcgmpg6fu"),
            ("datasync-25", "mdb57f180vupm0c7d9qs"),
            ("datasync-26", "mdbpj9m59fqq5nqqvr42"),
            ("datasync-27", "mdbf0olgfg9aspbbj0dh"),
            ("datasync-28", "mdbf4374fbb7eb1358rs"),
            ("datasync-29", "mdb7cvbvpenmbvt915hm"),
            ("datasync-30", "mdbi35lp70glc78nnn3v"),
            ("datasync-31", "mdba22grqt5p1g3f93a7"),
            ("datasync-32", "mdbk09krgi8gn0f430ml"),
            ("datasync-33", "mdb51450an4dsqklasjc"),
            ("datasync-34", "mdboat2eeucid77j3ree"),
            ("datasync-35", "mdb8rvsohppvgqgj7d79"),
            ("datasync-36", "mdb5k2js3f1lpl21gdh1"),
            ("datasync-37", "mdb6l8d7lh0tqnut7i0c"),
            ("datasync-38", "mdb2s97rlf951d4rkcu2"),
            ("datasync-39", "mdbrcc4g2boer1deo3d2"),
            ("datasync-40", "mdbd4hgfvma5qgmq9njh"),
            ("datasync-41", "mdbeh6np45v2s4dkmgmj"),
            ("datasync-42", "mdbifrocn5ql60vb7ser"),
            ("datasync-43", "mdb5j60n2kb4nfgsvofb"),
            ("datasync-44", "mdbbntjso8gl2ae64lft"),
            ("datasync-45", "mdbjbit55847begfcer4"),
            ("datasync-46", "mdbcdrr1ua67vqdp42aa"),
            ("datasync-47", "mdbvj98j0r9gkco3vcs4"),
            ("datasync-48", "mdb378e8cunq0p2sb0b9"),
            ("datasync-load-01", "mdbrsaf2hjnu0uguajeu"),
        ]:
            hosts[self.PSEUDO_CGROUP].append(mdb_host(*h))

        return hosts

    @cached(days=999)
    def collect_stats(self, fqdn, date):
        date = date_to_timestamp(date, round_to_day=True)
        results = {
            'date': date,
            'fqdn': fqdn.name,
        }
        for name, sig in self.signals.items():
            try:
                results[name] = self.get_sig(fqdn.id, sig, date)
            except RetryLimitExceeded:
                logger.warning('Signal "%s" is not collected from %s:%s' % (sig, fqdn.name, fqdn.id))
                return results
        return results


yasmcollector = YasmCollector()
yasmhwcollector = YasmHardwareCollector()
yasmpgcollector = YasmPGCollector()
yasmsharpeicollector = YasmSharpeiCollector()
yasmqloudcollector = YasmQloudCollector()
yasmmdbcollector = YasmMDBCollector()
yasmdeploycollector = YasmDeployCollector()
