from datetime import datetime, timedelta
from copy import deepcopy
from threading import RLock
import time
import logging
import functools


def retry(tries=10, delay=1, backoff=1.5):
    def wrapper(func):
        @functools.wraps(func)
        def inner(*args, **kwargs):
            _tries, _delay = tries, delay
            while _tries > 0:
                result = func(*args, **kwargs)
                if result is None:
                    _tries -= 1
                    time.sleep(_delay)
                    _delay *= backoff
                else:
                    return result
        return inner
    return wrapper


def cached(seconds=0, minutes=0, hours=1, days=0):
    time_delta = timedelta(seconds=seconds, minutes=minutes, hours=hours, days=days)

    def decorate(func):
        func._lock = RLock()
        func._updates = {}
        func._results = {}

        @functools.wraps(func)
        def do_cache(*args, **kwargs):

            lock = func._lock
            lock.acquire()

            try:
                key = (args, tuple(sorted(kwargs.items(), key=lambda i:i[0])))

                updates = func._updates
                results = func._results

                now = datetime.now()
                updated = updates.get(key, now)

                if key not in results or now - updated > time_delta:
                    updates[key] = now
                    result = func(*args, **kwargs)
                    results[key] = deepcopy(result)
                    return result
                else:
                    return deepcopy(results[key])
            finally:
                lock.release()
        return do_cache
    return decorate


def logger(level='debug', write_result=False):
    def decorator(func):
        @functools.wraps(func)
        def wrapper(*args, **kwargs):
            log = getattr(logging, level)
            try:
                log(
                    "Executing {func}({args}, {kwargs})".format(
                        func=func.__name__,
                        args=", ".join(["\"{}\"".format(arg) for arg in args]),
                        kwargs=", ".join(["{}=\"{}\"".format(str(key), str(val)) for key, val in kwargs.items()])
                    )
                )

                ret = func(*args, **kwargs)

                if write_result:
                    log("Result is: '{}'".format(str(ret)))

                return ret
            except Exception as e:
                err = "Exception in {}".format(func.__name__)
                logging.exception(err)

                raise e
        return wrapper
    return decorator
