import requests
from lib.decorators import retry
# import paramiko
import logging
import sys
import urllib3
urllib3.disable_warnings()

from datetime import date as dt_date
from datetime import datetime, timedelta
import time
from dateutil import parser
from collections import defaultdict
from decimal import Decimal

logging.basicConfig(level=logging.DEBUG, format="[%(asctime)s] %(levelname)s %(name)s %(message)s")
logger = logging.getLogger(__name__)

FIVE_M = 300
HOUR = 3600
DAY = 3600 * 24

@retry(2, 1.5, 1)
def get_url(url, to_json=True, auth=None, headers={}, timeout=300, raw=False,
            chunk_size=1024*2, astype=str, raise_on_status_code='502 503 504'):
    req = requests.get(url, headers=headers, stream=raw, auth=auth, timeout=timeout, verify=False)
    if 200 == req.status_code:
        if to_json:
            try:
                return req.json()
            except:
                pass
        content = req.raw.read(chunk_size) if raw else req.content
        ret = content.decode(req.encoding) if req.encoding else content.decode('utf8', "ignore")
        try:
            return astype(ret)
        except:
            return ret
    elif raise_on_status_code:
        if str(req.status_code) in raise_on_status_code:
            raise Exception(req.text, req.status_code)


def to_float(val):
    if val is None:
        return 0.0
    else:
        v = Decimal(val)
        if v.is_nan():
            return 0.0
        return float(v)


def get_nth_percentile(data, n):
    if n < 1:
        n = int(n*100)
    sorted_data = sorted(data)
    assert 0 <= n < 100
    index = (len(sorted_data) * n) // 100
    assert index < len(sorted_data)
    return sorted_data[index]

def date_to_timestamp(date, round_to_day=False, tz_delta=3600*3):
    if isinstance(date, (int, float)):
        res = int(date)
    elif isinstance(date, datetime):
        res = int(time.mktime(date.timetuple()))
        if not date.tzinfo and sys.version_info.major == 3:
            res += tz_delta
    elif isinstance(date, dt_date):
        res = int(time.mktime(date.timetuple()))
    elif isinstance(date, (str, basestring)):
        dt = parser.parse(date)
        res = time.mktime(dt.timetuple())
        if not dt.tzinfo:
            res += tz_delta
    else:
        raise TypeError('Unsupported date type: %s' % type(date))
    if round_to_day:
        res -= res % (3600*24)
    return res


class CTree(object):
    """
    Was supposed to be conductor adapter
    """
    _groups = {}
    _hosts = set()

    def __init__(self, name, childs=None, hosts=()):
        if name in self._groups:
            raise NameError('Name already exists')
        self.name = name
        childs = childs or {}
        self.childs = childs
        self.hosts = set()
        self.add_hosts(hosts)
        self._groups.update({self.name: self})

    @property
    def child_names(self):
        return [k for k in self.childs]

    def add_hosts(self, hosts=()):
        self.hosts.update(hosts)
        self._hosts.update(hosts)

    def add_child(self, path=(), hosts=()):
        if not path:
            raise Exception('Path not specified')

        if len(path) == 1:
            if path[0] in self.child_names:
                self.childs[path[0]].add_hosts(hosts)
            else:
                self.childs[path[0]] = CTree(path[0], hosts=hosts)
        else:
            if path[0] not in self.child_names:
                self.childs[path[0]] = CTree(path[0])
            self.childs[path[0]].add_child(path[1:], hosts=hosts)

    def _get_key_path(self, key, _path=()):
        if key == self.name or key in self.hosts:
            return _path + (self.name,)
        else:
            for k, c in self.childs.items():
                res = c._get_key_path(key, _path + (self.name,))
                if res:
                    return res
        return None

    def get_key_path(self, key):
        if not (key in self._groups or key in self._hosts):
            return None
        path = self._get_key_path(key)
        if not path:
            return None
        return path[1:]

    def get_subtree(self, key):
        path = self.get_key_path(key)
        if not path:
            return None

        curr_tree = self
        for c in path:
            curr_tree = curr_tree.childs[c]
        return curr_tree

    def get_groups(self, key=None, include=False):
        result = []
        if include:
            result.append(self.name)
        tree = self
        if key:
            tree = self.get_subtree(key)
        if not tree:
            return result

        groups = [g for g in tree.childs.values()]
        while groups:
            t_groups = []
            for g in groups:
                result.append(g.name)
                t_groups.extend(g.childs.values())
            groups = t_groups
        return result

    def get_hosts(self, key=None, include=False):
        groups = self.get_groups(key, include)
        hosts = []
        hosts.extend(self.hosts)
        for g in groups:
            tree = self.get_subtree(g)
            hosts.extend(tree.hosts)
        return hosts

    def is_child(self, child, parent):
        parent = self.get_subtree(parent)
        if not parent:
            return False
        if parent.get_subtree(child) or child in parent.get_hosts():
            return True
        return False


def reverse_dict(dct):
    res = defaultdict(set)
    for k, v in dct.items():
        if hasattr(v, '__iter__') and not isinstance(v, (str, basestring)):
            for el in v:
                res[el].update((k,))
        else:
            res[v].update((k,))
    return res


def yesterday():
    dt = datetime.now()
    return dt - timedelta(days=1)


def chunk_dates(sd, ed, chunk_size=30 * 24 * 3600, format='%Y-%m-%d'):
    cd = date_to_timestamp(sd)
    ed = date_to_timestamp(ed)
    date_chunks = []
    while(cd < ed):
        date_chunks.append((cd, min(cd + chunk_size, ed)))
        cd += chunk_size + 24 * 3600
    if format:
        date_chunks = [[datetime.fromtimestamp(d).strftime(format) for d in r] for r in date_chunks]
    return date_chunks
