# -*- coding: utf-8 -*-
import json

from tornado.gen import Return, coroutine
from tornado.httpclient import AsyncHTTPClient


class ConductorAPI(object):

    def __init__(self, api_url_ro, api_url_rw, token):
        self.api_url_ro = api_url_ro
        self.api_url_rw = api_url_rw
        self.token = token

    @property
    def headers(self):
        headers = {
            'Authorization': 'OAuth {token}'.format(token=self.token),
            'Accept': 'application/json',
            'Content-Type': 'application/json'
        }

        return headers

    @coroutine
    def hostlist(self, logger, group):
        result = None

        try:
            URL = '{api_url_ro}/{method}/{group}?format=json&recursive=no&fields=fqdn,root_datacenter_name'.format(
                api_url_ro=self.api_url_ro, method='groups2hosts', group=group
            )

            logger.debug('Fetching hosts from group %s' % group)
            client = AsyncHTTPClient()
            response = yield client.fetch(URL, method='GET', raise_error=False)

            if response.error:
                logger.error('Unable to fetch hosts from conductor => %s' % response.error.message)

            else:
                try:
                    result = json.loads(response.body)
                    logger.debug('Hostlist from group => %s was successfully retreived' % group)

                except Exception as e:
                    logger.critical('Unable to parse JSON from conductor => %s' % e.message)

        except Exception as e:
            logger.error('Uncaught exception while retreiving hostlist from conductor: %s' % e.message)

        finally:
            raise Return(result)

    @coroutine
    def create(self, logger, package):
        result = None

        try:
            URL = '{api_url_rw}/{method}'.format(api_url_rw=self.api_url_rw, method='hosts')

            logger.debug(
                'Creating conductor host {host} in group => {group}'.format(
                    host=package['host']['fqdn'],
                    group=package['host']['group']['name'],
                )
            )
            client = AsyncHTTPClient()
            response = yield client.fetch(
                URL, method='POST', body=json.dumps(package), headers=self.headers, raise_error=False
            )

            if response.error:
                logger.error('Unable to create host in conductor => %s' % response.error.message)

            else:
                try:
                    result = json.loads(response.body)
                    logger.debug('Host => %s was successfully created in conductor' % package['host']['fqdn'])

                except Exception as e:
                    logger.critical('Unable to parse JSON from conductor => %s' % e.message)

        except Exception as e:
            logger.error('Uncaught exception while creating host in conductor: %s' % e.message)

        finally:
            raise Return(result)

    @coroutine
    def delete(self, logger, hostname):
        result = None

        try:
            URL = '{api_url_rw}/{method}/{hostname}'.format(
                api_url_rw=self.api_url_rw, method='hosts', hostname=hostname
            )

            logger.debug('Removing host from conductor => %s' % hostname)
            client = AsyncHTTPClient()
            response = yield client.fetch(URL, method='DELETE', headers=self.headers, raise_error=False)

            if response.error:
                logger.error('Unable to delete host from conductor => %s' % response.error.message)

            else:
                result = True
                logger.debug('Host => %s was successfully deleted from conductor' % hostname)

        except Exception as e:
            logger.error('Uncaught exception while deleting host from conductor: %s' % e.message)

        finally:
            raise Return(result)
