# -*- coding: utf-8 -*-
from tornado.gen import Return, coroutine

from yp.client import YpClient
from yp.common import YP_PRODUCTION_CLUSTERS

from ..utils.misc import shorten_ipv6


class DeployAPI(object):

    def __init__(self, token):
        self.clients = {
            cluster: YpClient(cluster, config={"token": token})
            for cluster in YP_PRODUCTION_CLUSTERS
        }

    @coroutine
    def instancelist(self, environment):
        result = []
        try:
            environment.logger.debug('Fetching instances from YP')
            for podset in environment.podsets.keys():
                for cluster in self.clients.keys():
                    cluster_instancelist = yield self.cluster_instancelist(environment, cluster, podset)
                    result.extend(cluster_instancelist)

        except Exception as e:
            environment.logger.error('Uncaught exception while retrieving instances from YP: %s' % e.message)

        finally:
            raise Return(result)

    @coroutine
    def cluster_instancelist(self, environment, cluster, podset):
        result = []
        try:
            environment.logger.debug('Fetching instance list from YP %s, podset %s' % (cluster, podset))
            all_pods = self.clients[cluster].select_objects(
                'pod',
                filter='[/meta/pod_set_id] = "%s.%s"' % (environment.stage, podset),
                selectors=['/status/agent/pod_agent_payload/status'],
            )
            for pod in all_pods:
                for box in pod[0]['boxes']:
                    if box['id'] not in environment.podsets[podset]:
                        continue
                    result.append({
                        'id': box['id'],
                        'podId': pod[0]['id'],
                        'podSetId': podset,
                        'dc': cluster,
                        'target': '%s.%s.yp-c.yandex.net' % (pod[0]['id'], cluster),
                        'addresses': {'A': None, 'AAAA': shorten_ipv6(box['ip6_address'])},
                    })

        except Exception as e:
            environment.logger.error('Uncaught exception while retrieving instance list from YP %s, podset %s: %s'
                                         % (cluster, podset, e.message))

        finally:
            raise Return(result)
