# -*- coding: utf-8 -*-

import datetime as dt
import json
import time
import random

from tornado.gen import Return, coroutine
from tornado.httpclient import HTTPClient


class DomainsAPI(object):

    def __init__(self, api_url, api_primitives_url, token, login):
        self.api_primitives_url = api_primitives_url
        self.api_url = api_url
        self.login = login
        self.token = token
        self.cached_zones = {}
        self.cached_recordsets = {}
        self.next_call = dt.datetime.now()

    @property
    def http_options(self):
        options = dict(
            headers={
                'X-Auth-Token': self.token,
                'Accept': 'application/json',
                'Content-Type': 'application/json'
            },
        )

        return options

    def __do_call(self, environment, url, **kwargs):
        client = HTTPClient()
        environment.logger.debug('_do_call to %s' % url)
        response = client.fetch(url, **kwargs)
        environment.logger.debug('_do_call to %s: got response code %d' % (url, response.code))
        return response

    def http_call(self, environment, url, **kwargs):
        environment.logger.info('Starting http_call to %s' % url)
        now = dt.datetime.now()
        while now < self.next_call:
            environment.logger.info('Waiting for http_call to %s' % url)
            time.sleep((self.next_call - now).total_seconds()+3)
            now = dt.datetime.now()

        self.next_call = dt.datetime.now() + dt.timedelta(seconds=3)
        environment.logger.info('Doing actual http_call to %s' % url)
        response = self.__do_call(environment, url, **kwargs)
        try_count = 1
        while response.code == 429 and try_count < 5:
            environment.logger.info('Got 429 for http_call to %s' % url)
            # try again with sleep, maybe we'll have luck...
            time.sleep(3+random.uniform(-2, 2))
            self.next_call = dt.datetime.now() + dt.timedelta(seconds=3)
            environment.logger.info('Trying again http_call to %s' % url)
            try_count = try_count + 1
            response = self.__do_call(environment, url, **kwargs)
            environment.logger.debug('Got response for %s: %d' % (url, response.code))

        environment.logger.info('Got response for http_call to %s' % url)
        return response

    @coroutine
    def recordset(self, environment, uuid, limit=100, offset=0):
        result = None
        cached = self.cached_recordsets.get(uuid, None)
        if cached is not None and cached[1] > dt.datetime.now():
            environment.logger.info("Using cached result for recordset %s" % uuid)
            result = cached[0]
        else:
            try:
                response = yield self.records(environment, uuid, limit, offset)
                recordset = list()

                if isinstance(response, tuple):
                    entries, records = response
                    recordset.extend(records)

                    while not len(recordset) == entries:
                        offset = offset + limit

                        if offset >= 20000:
                            environment.logger.error('There so much record found in zone! More than 200k! Stop it!')
                            break

                        response = yield self.records(environment, uuid, limit, offset)

                        if not isinstance(response, tuple):
                            # :( received some shit => log message must be thrown via self.records
                            break

                        entries, records = response
                        recordset.extend(records)

                    if len(recordset) == entries:
                        result = recordset
                environment.logger.debug('Recordset successfully fetched from DNS server.')
                environment.logger.info('Saving recordset %s to cache' % uuid)
                self.cached_recordsets[uuid] = [result, (dt.datetime.now() + dt.timedelta(minutes=15))]
            except Exception as e:
                environment.logger.error('Uncaught exception while fetching partial DNS records: %s: %s' % (e.__class__, e.message))
                if uuid in self.cached_recordsets:
                    del self.cached_recordsets[uuid]

        raise Return(result)

    @coroutine
    def records(self, environment, uuid, limit, offset):
        result = None

        try:
            URL = '{api_url}/{login}/zones/{uuid}/records?showRecords?limit={limit}&offset={offset}'.format(
                api_url=self.api_url, login=self.login, uuid=uuid, limit=limit, offset=offset
            )

            response = self.http_call(environment, URL, method='GET', raise_error=False, **self.http_options)

            if response.error:
                environment.logger.error('Unable to get records from DNS server => %s' % response.error.message)
            else:
                environment.logger.info('Saving records for %s to cache' % uuid)

            try:
                body = json.loads(response.body)
                entries = body['zones'][0]['recordsList']['totalEntries']
                records = body['zones'][0]['recordsList']['records']

                result = entries, records
            except Exception as e:
                environment.logger.critical('Unable to parse partial JSON from DNS => %s' % e.message)

        except Exception as e:
            environment.logger.error('Uncaught exception while fetching partial DNS records: %s: %s' % (e.__class__, e.message))

        finally:
            raise Return(result)

    @coroutine
    def zoneinfo(self, environment, zone):
        result = None

        try:
            URL = '{api_url}/{login}/zones?{zone}'.format(
                api_url=self.api_url, login=self.login, zone=zone
            )

            cached = self.cached_zones.get(zone, None)
            if cached is not None and cached[1] > dt.datetime.now():
                environment.logger.info('Using cached result for zone %s' % zone)
                response = cached[0]
            else:
                response = self.http_call(environment, URL, method='GET', raise_error=False, **self.http_options)
                if response.error:
                    environment.logger.error('Unable to get zone info from DNS server => %s' % response.error.message)
                    if zone in self.cached_zones:
                        del self.cached_zones[zone]
                else:
                    environment.logger.info('Saving to zone %s to cache' % zone)
                    self.cached_zones[zone] = [response, (dt.datetime.now() + dt.timedelta(minutes=15))]

            try:
                result = json.loads(response.body)
                environment.logger.debug('Zone info successfully fetched from DNS server.')

            except Exception as e:
                environment.logger.critical('Unable to parse JSON from DNS => %s' % e.message)

        except Exception as e:
            environment.logger.error('Uncaught exception while submitting DNS primitives: %s' % e.message)

        finally:
            raise Return(result)

    @staticmethod
    def primitive(name, content, rtype='CNAME', ttl=60, operation='add'):
        _primitive = {
            'name': name,
            'data': content,
            'type': rtype,
            'ttl': int(ttl),
            'operation': operation
        }

        return _primitive

    @coroutine
    def submit(self, environment, primitives):
        result = None

        try:
            URL = '{api_primitives_url}/{login}/primitives?showOperations'.format(
                api_primitives_url=self.api_primitives_url, login=self.login
            )

            primitives = json.dumps({'primitives': primitives})

            response = self.http_call(environment, URL, method='PUT', body=primitives, raise_error=False, **self.http_options)

            if response.error:
                environment.logger.error('Unable to submit primitives to DNS server => %s' % response.error.message)

            else:
                try:
                    result = json.loads(response.body)
                    environment.logger.debug('DNS primitives was submitted. Awaiting result to return')

                except Exception as e:
                    environment.logger.critical('Unable to parse JSON from DNS => %s' % e.message)

        except Exception as e:
            environment.logger.error('Uncaught exception while submitting DNS primitives: %s' % e.message)

        finally:
            raise Return(result)
