# -*- coding: utf-8 -*-
from tornado.gen import Return, coroutine

from ..utils.logger import EnvironmentLogger
from ..utils.misc import with_retry


class QloudEnvironment(object):

    def __init__(self, package, qloud_api):
        self.environment_id = package['environmentId']
        self.status = package['status']
        self.version = package['version']
        self.comment = package.get('comment', 'NO COMMENTS. Shame!')
        self.author = ''
        self.qloud_api = qloud_api
        self.url = self.qloud_api.generate_url(self.environment_id, self.version)
        self.logger = EnvironmentLogger(self.environment_id, self.status)
        self.project, self.cluster, self.environment = self.environment_id.split('.')
        self._is_external_activation = None

    def is_qloud_deactivation(self):
        service_comments = {
            'Hosts deactivated: ',
            'Reallocation due to hosts maintenance',
        }
        for service_comment in service_comments:
            if self.comment.startswith(service_comment):
                return True
        return False

    @coroutine
    def get_author(self):
        if not self.author:
            if self.is_qloud_deactivation():
                self.author = 'QloudMaintenance'
            else:
                self.author = yield with_retry(self.qloud_api.get_author, (self, ))
        raise Return(self.author)

    @coroutine
    def is_external_activation(self):
        if self._is_external_activation is None:
            environment_activation_recipe = yield with_retry(
                self.qloud_api.get_activation_recipe, (self,),
            )
            self._is_external_activation = environment_activation_recipe == 'EXTERNAL'
        raise Return(self._is_external_activation)

    def is_supported(self, cfg):
        if self.project not in cfg:
            self.logger.info('Unsupported project')
            return False
        if self.cluster not in cfg[self.project]:
            self.logger.info('Unsupported cluster')
            return False
        if self.environment not in cfg[self.project][self.cluster]:
            self.logger.info('Unsupported environment')
            return False

        return True


class DeployEnvironment(object):

    def __init__(self, stage, podsets):
        self.stage = stage
        self.url = "https://deploy.yandex-team.ru/project/%s" % stage
        self.logger = EnvironmentLogger(stage, 'SYNC')
        self.podsets = podsets

    def is_supported(self):
        if self.podsets is None:
            self.logger.info('Unsupported stage')
            return False
        return True
