# -*- coding: utf-8 -*-
import datetime
import json
import random
import re

from tornado.gen import Return, coroutine, sleep
from tornado.httpclient import AsyncHTTPClient


class InfraAPI(object):

    def __init__(self, api_url, token, service_id, environment_id, notified_environments):
        self.api_url = api_url
        self.token = token
        self.service_id = service_id
        self.environment_id = environment_id
        self.event_type = 'maintenance'
        self.severity = 'minor'
        self.notified_environments = notified_environments
        self.ticket_re = re.compile(r'(CHEMODAN-\d+)')

    def is_notification_enabled(self, environment):
        if environment.environment_id.rsplit('.', 1)[-1] not in self.notified_environments:
            return False

        return not environment.is_qloud_deactivation()

    @property
    def headers(self):
        return {
            'Authorization': 'OAuth {token}'.format(token=self.token),
            'Accept': 'application/json',
            'Content-Type': 'application/json'
        }

    @coroutine
    def notify_deploy(self, environment):
        if not self.is_notification_enabled(environment):
            environment.logger.info('We create infra event only when your app is in stable')
            return

        result = None

        # Sleep random time to wait for second hook
        yield sleep(random.randint(1, 30))

        current_event_ids = yield self.get_current_event_ids(environment)
        if current_event_ids:
            raise Return(result)

        now = int(datetime.datetime.now().strftime('%s'))
        author = yield environment.get_author()
        event = {
            'serviceId': self.service_id,
            'environmentId': self.environment_id,
            'type': self.event_type,
            'severity': self.severity,
            'setAllAvailableDc': True,
            'sendEmailNotifications': True,
            'title': environment.environment_id,
            'description': u'author: %s\n%s\n\nurl: %s' % (author, environment.comment, environment.url),
            'startTime': int(datetime.datetime.now().strftime('%s')),
            'finishTime': now + 3600,
        }

        ticket_ids = self.ticket_re.findall(environment.comment)
        if ticket_ids:
            event['tickets'] = ", ".join(ticket_ids)

        try:
            environment.logger.debug('Create infra event for %s: %s' % (environment.environment_id, json.dumps(event)))
            client = AsyncHTTPClient()
            response = yield client.fetch(
                '%s/events' % self.api_url,
                method='POST',
                body=json.dumps(event),
                headers=self.headers,
                raise_error=False,
            )

            if response.error:
                environment.logger.error('Unable to create event in infra => %s' % response.error)
            else:
                try:
                    result = json.loads(response.body)
                    environment.logger.debug('Event for %s was successfully retreived' % environment.environment_id)

                except Exception as e:
                    environment.logger.critical('Unable to parse JSON from infra => %s' % e.message)

        except Exception as e:
            environment.logger.error('Uncaught exception while creating event in infra: %s' % e.message)

        finally:
            raise Return(result)

    @coroutine
    def get_current_event_ids(self, environment):
        result = None

        try:
            URL = (
                '%(url)s/events/current?'
                'type=%(type)s&severity=%(severity)s&serviceId=%(serviceId)s&environmentId=%(environmentId)s'
                % dict(
                    url=self.api_url,
                    type=self.event_type,
                    severity=self.severity,
                    serviceId=self.service_id,
                    environmentId=self.environment_id,
                )
            )
            environment.logger.debug('Get infra events')
            client = AsyncHTTPClient()
            response = yield client.fetch(URL, method='GET', headers=self.headers, raise_error=False)

            if response.error:
                environment.logger.error('Unable to get events from infra => %s' % response.error.message)

            else:
                try:
                    events_list = json.loads(response.body)
                    environment.logger.debug('Events was successfully got from infra')

                except Exception as e:
                    environment.logger.critical('Unable to parse JSON from infra => %s' % e.message)
                else:
                    result = [
                        e['id']
                        for e in events_list
                        if e['title'] == environment.environment_id and environment.url in e['description']
                    ]
        except Exception as e:
            environment.logger.error('Uncaught exception while fetching events from infra: %s' % e.message)

        finally:
            raise Return(result)

    @coroutine
    def resolve_event(self, environment):
        result = []

        current_event_ids = yield self.get_current_event_ids(environment)
        if not current_event_ids:
            raise Return(result)

        event_diff = {
            'finishTime': int(datetime.datetime.now().strftime('%s')),
            'sendEmailNotifications': True,
            'resolve': True,
        }

        try:
            for current_event_id in current_event_ids:
                environment.logger.debug('Resolve event => %s' % current_event_id)
                client = AsyncHTTPClient()
                response = yield client.fetch(
                    '%s/events/%s' % (self.api_url, current_event_id),
                    method='PUT',
                    body=json.dumps(event_diff),
                    headers=self.headers,
                    raise_error=False,
                )

                if response.error:
                    environment.logger.error('Error while resolve event => %s' % response.error.message)

                else:
                    try:
                        result.append(json.loads(response.body))
                        environment.logger.debug('Event %s was successfully resolved' % current_event_id)
                    except Exception as e:
                        environment.logger.critical('Unable to parse JSON from infra => %s' % e.message)

        except Exception as e:
            environment.logger.error('Uncaught exception while resolving event in infra: %s' % e.message)

        finally:
            raise Return(result)
