# -*- coding: utf-8 -*-
import json
from urlparse import urljoin

from tornado.gen import Return, coroutine
from tornado.httpclient import AsyncHTTPClient

from ..utils.misc import shorten_ipv6


class QloudAPI(object):

    def __init__(self, api_url, token):
        self.api_url = api_url
        self.token = token

    def generate_url(self, environment, version):
        return urljoin(self.api_url, '/projects/' + environment.replace('.', '/') + "?version=%s" % version)

    @coroutine
    def instancelist(self, environment):
        result = None
        try:
            environment.logger.debug('Fetching instances from QLOUD')

            URL = '{api_url}/status/{environment_id}'.format(
                api_url=self.api_url,
                environment_id=environment.environment_id,
            )
            headers = {
                'Authorization': 'OAuth %s' % self.token,
                'Accept': 'application/json',
                'Content-Type': 'application/json'
            }

            client = AsyncHTTPClient()
            response = yield client.fetch(URL, method='GET', raise_error=False, headers=headers)

            if response.error:
                environment.logger.error('Unable to fetch instances from qloud => %s' % response.error.message)

            else:
                try:
                    result = json.loads(response.body)
                    environment.logger.debug('Instances from QLOUD was successfully retreived')

                except Exception as e:
                    environment.logger.critical('Unable to parse JSON from QLOUD => %s' % e.message)

        except Exception as e:
            environment.logger.error('Uncaught exception while retreiving instances from QLOUD: %s' % e.message)

        finally:
            raise Return(result)

    @coroutine
    def plain_instance_list(self, environment):
        result = None
        try:
            environment.logger.debug('Fetching plain instance list from QLOUD')

            URL = '{api_url}/instances/{environment_id}/{version}'.format(
                api_url=self.api_url, environment_id=environment.environment_id, version=environment.version
            )
            headers = {
                'Authorization': 'OAuth %s' % self.token,
                'Accept': 'application/json',
                'Content-Type': 'application/json'
            }

            client = AsyncHTTPClient()
            response = yield client.fetch(URL, method='GET', raise_error=False, headers=headers)

            if response.error:
                environment.logger.error('Unable to plain fetch instances from qloud => %s' % response.error.message)

            else:
                try:
                    result = json.loads(response.body)
                    environment.logger.debug('Plain instance list from QLOUD was successfully retreived')

                except Exception as e:
                    environment.logger.critical('Unable to parse JSON from QLOUD => %s' % e.message)

        except Exception as e:
            environment.logger.error(
                'Uncaught exception while retreiving plain instance list from QLOUD: %s' % e.message)

        finally:
            raise Return(result)

    @coroutine
    def component_addresses(self, plain_instance_list, component_id):
        for instance in plain_instance_list:
            if instance.get('instanceId') == component_id:
                raise Return({
                    'AAAA': shorten_ipv6(instance.get('instanceIp')),
                    'A': instance.get('instanceIpv4'),
                })
        else:
            raise KeyError('%s not found in plain_instance_list' % component_id)

    @coroutine
    def get_author(self, environment):
        result = None
        try:
            environment.logger.debug('Fetching environment info from QLOUD')

            url = '{api_url}/environment/status/{environment_id}/{version}'.format(
                api_url=self.api_url,
                environment_id=environment.environment_id,
                version=environment.version,
            )
            headers = {
                'Authorization': 'OAuth %s' % self.token,
                'Accept': 'application/json',
                'Content-Type': 'application/json'
            }

            client = AsyncHTTPClient()
            response = yield client.fetch(url, method='GET', raise_error=False, headers=headers)

            if response.error:
                environment.logger.error('Unable to get environment status from qloud => %s' % response.error.message)

            else:
                try:
                    result = json.loads(response.body)['author']
                    environment.logger.debug('Status from QLOUD was successfully retreived')

                except Exception as e:
                    environment.logger.critical('Unable to parse JSON from QLOUD => %s' % e.message)

        except Exception as e:
            environment.logger.error('Uncaught exception while retreiving status from QLOUD: %s' % e.message)

        finally:
            raise Return(result)

    @coroutine
    def get_activation_recipe(self, environment):
        result = None
        try:
            environment.logger.debug('Fetching environment dump from QLOUD')

            url = '{api_url}/environment/dump/{environment_id}/{version}'.format(
                api_url=self.api_url,
                environment_id=environment.environment_id,
                version=environment.version,
            )
            headers = {
                'Authorization': 'OAuth %s' % self.token,
                'Accept': 'application/json',
                'Content-Type': 'application/json'
            }

            client = AsyncHTTPClient()
            response = yield client.fetch(url, method='GET', raise_error=False, headers=headers)

            if response.error:
                environment.logger.error('Unable to get environment dump from qloud => %s' % response.error.message)

            else:
                try:
                    components = json.loads(response.body)['components']
                    environment.logger.debug('Components dump from QLOUD was successfully retrieved')

                except Exception as e:
                    environment.logger.critical('Unable to parse JSON from QLOUD => %s' % e.message)
                else:
                    is_external = 0
                    for component in components:
                        recipe = component.get('activateRecipe', {}).get('recipe')
                        is_external |= recipe == 'EXTERNAL'
                        if is_external:
                            break
                    result = 'EXTERNAL' if is_external else 'INTERNAL'
        except Exception as e:
            environment.logger.error('Uncaught exception while retrieving dump from QLOUD: %s' % e.message)

        finally:
            raise Return(result)
