# -*- coding: utf-8 -*-
import json

from tornado.gen import coroutine, Return
from tornado.httpclient import AsyncHTTPClient

from ..utils.logger import tskv_escape


class TelegramAPI(object):

    def __init__(self, token, chat_id, environments):
        self.token = token
        self.hook_url = 'https://api.telegram.org/bot%s/sendMessage'
        self.environments = environments
        self.body = {
            'chat_id': chat_id,
            'text': '',
            'parse_mode': 'HTML',
        }
        self.headers = {
            'Accept': 'application/json',
            'Content-Type': 'application/json'
        }
        self.deploy_message_template = (
            "%(emoji)s <b>%(environment_id)s</b> <a href=\\\"%(environment_url)s\\\">%(status)s</a>\n"
            "@%(environment_author)s: %(environment_comment)s"
        )
        self.status = {
            'ALLOCATING': u"🚀",
            'DEPLOYED': u"✅",
            'FAILED': u"❌",
        }

    def is_notification_enabled(self, environment):
        return environment in self.environments

    def is_notification_needed(self, status):
        return status in self.status

    @coroutine
    def send_message(self, logger, message, chat_id=None, token=None):
        if token is None:
            token = self.token
        body = dict(self.body, text='%s')
        if chat_id is not None:
            body['chat_id'] = chat_id

        body = json.dumps(body) % message

        try:
            logger.debug('Notify Telegram => %s' % tskv_escape(unicode(body)))
            client = AsyncHTTPClient()
            response = yield client.fetch(
                self.hook_url % token,
                method='POST',
                body=body,
                headers=self.headers,
                raise_error=False
            )
        except Exception as e:
            logger.error('Uncaught exception while notifying Telegram: %s' % e.message)
        else:
            if response.error:
                logger.error('Unable to notify Telegram => %s' % tskv_escape(response.error.message))
            else:
                if 'ok' in response.body:
                    logger.info('Telegram notified successfully')
                else:
                    logger.critical('Telegram was not accept our message')
                raise Return(response.body)

    @coroutine
    def notify(self, environment):
        if not self.is_notification_needed(environment.status):
            environment.logger.info('We notifying Telegram only when your deploy started or finished')
            raise Return()
        if not self.is_notification_enabled(environment.environment):
            environment.logger.info('We notifying Telegram only when your app in production')
            raise Return()

        comment = (
            json.dumps(environment.comment)[1:-1]
            .replace('&', '&amp;')
            .replace("\"", "&quot;")
            .replace("'", "&apos;")
            .replace(">", "&gt;")
            .replace("<", "&lt;")
        )

        external_activation = yield environment.is_external_activation()
        if external_activation:
            comment += ' <b>EXTERNAL activation</b>'

        if environment.is_qloud_deactivation() and not external_activation:
            environment.logger.info('We are not notifying Telegram about qloud maintenance')
            raise Return()

        author = yield environment.get_author()
        message = self.deploy_message_template % {
            'environment_id': environment.environment_id,
            'environment_url': environment.url,
            'environment_author': author,
            'environment_comment': comment,
            'emoji': self.status[environment.status],
            'status': environment.status.lower(),
        }
        yield self.send_message(environment.logger, message)
