# -*- coding: utf-8 -*-
from tornado.escape import json_encode, utf8
from tornado.gen import Return, coroutine
from tornado.web import RequestHandler

from ..utils.errors import CONTENT_TYPE_JSON_REQUIRED, INTERNAL_SERVER_ERROR
from ..utils.logger import logger
from ..utils.misc import with_retry


class BaseHandler(RequestHandler):

    def initialize(self):
        """
        You could pass some variables to this method when
        creating ``handlers`` in `disk.admin.conductor.sync.scripts.main:create_app`
        """

    def prepare(self):
        """
           ``prepare`` called each time, when we
           calling any API endpoint. Use it with honor, dude.
        """

        if 'application/json' not in self.request.headers.get('Content-Type', []):
            self.write_error(400, message=CONTENT_TYPE_JSON_REQUIRED)

    def write_error(self, status_code, message=INTERNAL_SERVER_ERROR, reason=None, **kwargs):
        self.set_status(status_code, reason=message)
        self.write({'error': message, 'code': status_code})

        logger.critical('Request interrupted due error => %s' % (reason or message))

    def write(self, *args, **kwargs):
        return self.response(*args, **kwargs)

    def response(self, response, encode=True):
        """Preparing result for compatible with tornado internals, nothing more"""
        self.set_header('Content-Type', 'application/json; charset=UTF-8')
        self.set_header('Cache-Control', 'no-store, no-cache, must-revalidate, max-age=0')

        try:
            if encode:
                package = json_encode(response)
            else:
                package = response

        except Exception as e:
            logger.error('Unhandled exception in main response method: %s' % e.message)
            self.write_error(500, message=e)

        else:
            package = utf8(package)
            self._write_buffer.append(package)

    @coroutine
    def generate_primitives(self, environment):
        raise Return(([], [], [], []))

    @coroutine
    def remove_from_dns(self, environment, primitives):
        if not primitives:
            raise Return()
        environment.logger.info('Trying to remove staled data from DNS')
        response = yield with_retry(
            self.application.domains_api.submit, (environment, primitives),
            tries=self.application.tries, timeout=self.application.timeout,
        )
        if not response:
            raise Return()

        environment.logger.info('Staled DNS records successfully removed')
        for zone, records in response['operations'].iteritems():
            for record in records:
                if record['valid'] == 'yes' and record['acl-match'] == 'yes':
                    environment.logger.info('Successfully removed record => %s' % record['name'])
                else:
                    environment.logger.error('Unable to remove record => %s' % record['name'])

    @coroutine
    def add_to_dns(self, environment, primitives):
        if not primitives:
            raise Return()
        environment.logger.info('Trying to add fresh data to DNS')
        response = yield with_retry(
            self.application.domains_api.submit, (environment, primitives),
            tries=self.application.tries, timeout=self.application.timeout,
        )
        if not response:
            raise Return()

        environment.logger.info('DNS records successfully refreshed')
        for zone, records in response['operations'].iteritems():
            for record in records:
                if record['valid'] == 'yes' and record['acl-match'] == 'yes':
                    environment.logger.info('Successfully created record => %s' % record['name'])
                else:
                    environment.logger.error('Unable to create record => %s' % record['name'])

    @coroutine
    def remove_from_conductor(self, environment, primitives):
        if not primitives:
            raise Return()
        environment.logger.info('Trying to remove staled hosts from conductor')

        for host in primitives:
            yield with_retry(
                self.application.conductor_api.delete, (environment.logger, host),
                tries=self.application.tries, timeout=self.application.timeout,
            )
            environment.logger.info('Successfully removed %s from conductor' % host)

    @coroutine
    def add_to_conductor(self, environment, primitives):
        if not primitives:
            raise Return()
        environment.logger.info('Trying to add new hosts to conductor')

        for package in primitives:
            yield with_retry(
                self.application.conductor_api.create, (environment.logger, package),
                tries=self.application.tries, timeout=self.application.timeout,
            )
            environment.logger.info('Successfully created %s in conductor' % package['host']['fqdn'])

    @coroutine
    def synchronize_cluster(self, environment):
        _add_to_conductor, _remove_from_conductor, _add_to_dns, _remove_from_dns = yield self.generate_primitives(
            environment
        )

        environment.logger.info('Apply primitives')
        # syncing data with DNS zones and Conductor groups
        for func, data in ((self.remove_from_dns, _remove_from_dns),
                           (self.add_to_dns, _add_to_dns),
                           (self.remove_from_conductor, _remove_from_conductor),
                           (self.add_to_conductor, _add_to_conductor)):
            yield func(environment, data)
        environment.logger.info('Syncing complete. Have fun, my dear.')

    @coroutine
    def fetch_conductor(self, environment, conductor_group):
        hostlist = yield with_retry(
            self.application.conductor_api.hostlist, (environment.logger, conductor_group),
            tries=self.application.tries, timeout=self.application.timeout,
        )
        if hostlist is None:
            raise Return()

        # reformat hostlist
        _hostlist = dict()
        for host in hostlist:
            _hostlist[host['fqdn']] = host

        raise Return(_hostlist)

    @coroutine
    def fetch_dns_zone(self, environment, subdomain, dns_zone):
        zoneinfo = yield with_retry(
            self.application.domains_api.zoneinfo, (environment, dns_zone),
            tries=self.application.tries, timeout=self.application.timeout,
        )
        if zoneinfo is None:
            raise Return()

        # ok, zoneinfo fetched, continue getting records list
        if not zoneinfo.get('zones'):
            environment.logger.error('Malformed zoneinfo received from DNS. Stopping')
            raise Return()

        info = zoneinfo['zones'].pop()

        # get records list from DNS for this component
        records = yield with_retry(
            self.application.domains_api.recordset, (environment, info['uuid']),
            tries=self.application.tries, timeout=self.application.timeout,
        )
        if records is None:
            raise Return()

        # reformat dns records
        suffix = u'%s.%s' % (subdomain, dns_zone)

        result = {}
        for record in records:
            if suffix in record['left-side']:
                result[record['left-side'].rstrip('.') + record['type']] = {
                    'left-side': record['left-side'].rstrip('.'),
                    'right-side': record['right-side'].rstrip('.'),
                    'type': record['type'],
                    'ttl': record['ttl']
                }
        raise Return(result)
