# -*- coding: utf-8 -*-
import json
from collections import defaultdict

from tornado.gen import Return, coroutine
from tornado.web import Finish

from .base import BaseHandler
from ..core.environment import DeployEnvironment
from ..utils.misc import with_retry, apply_async
from ..utils.logger import logger


class DeployHandler(BaseHandler):

    @coroutine
    def post(self):
        try:
            try:
                package = json.loads(self.request.body)
                logger.debug(package)

            except Exception as e:
                message = 'Unable to parse JSON from body => %s' % e.message
                logger.error(message)
                self.write_error(400, message=message)
                raise Finish()

            if 'stage' not in package:
                message = 'Malformed JSON body received. Is there is a hook from Deploy?'
                logger.critical(message)
                self.write_error(400, message=message)
                raise Finish()

            yield self.handle_stage(package['stage'])

        except Finish:
            pass

        except Exception as e:
            logger.error('Unhandled exception in main Deploy Handler: %s' % e.message)
            self.write_error(500)

        finally:
            self.finish()

    @coroutine
    def generate_primitives(self, environment):
        dns_zones, instances, conductor_groups = yield self.fetch_data(environment)
        # now we working w/ conductor hosts and compiled instance names from Deploy
        # containers for syncing w/ conductor and DNS
        _add_to_conductor = list()
        _remove_from_conductor = list()

        _add_to_dns = list()
        _remove_from_dns = list()

        # Searching for new hosts thats are should be added to conductor group
        for fqdn, instance in instances.items():
            if fqdn in conductor_groups[instance['group']]:
                environment.logger.debug(
                    '%s already in conductor group => %s. Skipping' % (fqdn, instance['group'])
                )
            else:
                conductor_host = {
                    'host': {
                        'fqdn': fqdn,
                        'datacenter_name': instance['root_datacenter_name'].split('-')[0],
                        'description': 'ssh endpoint for Deploy instance %s' % instance['target'],
                        'short_name': fqdn,
                        'group': {
                            'name': instance['group'],
                        }
                    }
                }
                _add_to_conductor.append(conductor_host)
                environment.logger.info(
                    'Missing %s in conductor => %s. Should be added!' % (fqdn, instance['group'])
                )

            # def primitive(self, name, content, type='CNAME', ttl=60, operation='add'):
            # check to what to do with this records = add/remove/update in DNS?
            if not instance['zone']:
                continue

            if fqdn + 'AAAA' not in dns_zones[instance['zone']]:
                environment.logger.info('Found missing DNS record for => %s' % fqdn)
                if instance['addresses']:
                    for rtype in ('AAAA', 'A'):
                        target = instance['addresses'][rtype]
                        if target:
                            primitive = self.application.domains_api.primitive(fqdn, target, rtype=rtype)
                            _add_to_dns.append(primitive)
                else:
                    continue

            else:
                if instance['addresses']:
                    records = (
                        dns_zones[instance['zone']][fqdn + 'AAAA'],
                        dns_zones[instance['zone']].get(fqdn + 'A'),
                    )
                else:
                    continue

                instance_add_to_dns = dict()
                for record in records:
                    if record is None:
                        continue

                    if ((record['type'] == 'CNAME' and record['right-side'] != instance['target']) or
                            (record['type'] == 'AAAA' and record['right-side'] != instance['addresses']['AAAA']) or
                            (record['type'] == 'A' and record['right-side'] != instance['addresses']['A'])):
                        # remove record first
                        primitive = self.application.domains_api.primitive(
                            fqdn, record['right-side'], rtype=record['type'],
                            ttl=record['ttl'], operation='delete'
                        )
                        _remove_from_dns.append(primitive)

                        if instance['addresses']:
                            for rtype in ('AAAA', 'A'):
                                target = instance['addresses'][rtype]
                                if target:
                                    instance_add_to_dns[rtype] = self.application.domains_api.primitive(
                                        fqdn, target, rtype=rtype
                                    )
                        else:
                            continue

                        environment.logger.info('Found changed DNS record => %s. Recreating it.' % fqdn)
                if instance_add_to_dns:
                    _add_to_dns.extend(instance_add_to_dns.values())

        # Searching for hosts that should be removed from conductor
        for group, hosts in conductor_groups.iteritems():
            for host, info in hosts.iteritems():
                if host not in instances:
                    environment.logger.info('Found staled host => %s that should be removed from conductor' % host)
                    _remove_from_conductor.append(host)
                    # searching for host in dns_zones

                    _host_in_dns = list()
                    for zone, zone_info in dns_zones.iteritems():
                        for rtype in ('CNAME', 'A', 'AAAA'):
                            host_rtype = host + rtype
                            if host_rtype in zone_info:
                                _host_in_dns.append(zone_info[host_rtype])

                    if not _host_in_dns:
                        environment.logger.debug('Given host => %s not found in DNS. Skipping.' % host)
                        continue

                    for record in _host_in_dns:
                        primitive = self.application.domains_api.primitive(
                            record['left-side'],
                            record['right-side'],
                            rtype=record['type'],
                            ttl=record['ttl'],
                            operation='delete'
                        )
                        _remove_from_dns.append(primitive)
                else:
                    if instances[host]['group'] != group:
                        environment.logger.info(
                            'There host => %s that should change his group => %s' % (host, info['group'])
                        )
                        _remove_from_conductor.append(host)

                        instance = instances[host]
                        conductor_host = {
                            'host': {
                                'fqdn': host,
                                'datacenter_name': instance['root_datacenter_name'],
                                'description': 'ssh endpoint for Deploy instance %s' % instance['target'],
                                'short_name': host,
                                'group': {
                                    'name': instance['group']
                                }
                            }
                        }
                        _add_to_conductor.append(conductor_host)
        raise Return((_add_to_conductor, _remove_from_conductor, _add_to_dns, _remove_from_dns))

    @coroutine
    def fetch_data(self, environment):
        subdomain = self.application.config['settings']['dns_subdomain_deploy']
        # containers for important data!
        dns_zones = defaultdict(dict)
        instances = defaultdict(dict)
        conductor_groups = defaultdict(dict)

        instance_list = yield with_retry(
            self.application.deploy_api.instancelist, (environment,),
            tries=self.application.tries, timeout=self.application.timeout,
        )
        if instance_list is None:
            raise Return()

        for instance in instance_list:
            component = instance['podSetId']
            if component not in environment.podsets:
                environment.logger.error('Pod Set %s not found in env config' % component)
                continue

            box_config = environment.podsets[component][instance['id']]
            dns_zone = box_config.get('dns_zone')
            if 'dns_record_format' in box_config and dns_zone:
                # cast Deploy pod format to our internals
                instance_fqdn = box_config['dns_record_format'].format(
                    box_id=instance['id'],
                    pod_id=instance['podId'],
                    dc=instance['dc'].lower(),
                    subdomain=subdomain,
                    zone=dns_zone,
                ).lower()
            else:
                continue

            box_conductor_group = box_config['conductor_group']
            instances[instance_fqdn] = {
                'fqdn': instance_fqdn,
                'target': instance['target'],
                'addresses': instance['addresses'],
                'group': box_conductor_group,
                'root_datacenter_name': instance['dc'].lower(),
                'zone': dns_zone
            }

            if dns_zone and dns_zone not in dns_zones:
                # get zone uuid and its records from DNS
                zone_dict = yield self.fetch_dns_zone(environment, subdomain, dns_zone)
                if zone_dict is None:
                    raise Return()
                dns_zones[dns_zone] = zone_dict

            if box_conductor_group not in conductor_groups:
                # get hosts from conductor group for given component
                _hostlist = yield self.fetch_conductor(environment, box_conductor_group)
                if _hostlist is None:
                    raise Return()
                conductor_groups[box_conductor_group] = _hostlist

        raise Return((dns_zones, instances, conductor_groups))

    @coroutine
    def handle_stage(self, stage):
        try:
            environment = DeployEnvironment(stage, self.application.config['deploy_clusters'].get(stage))
            if not environment.is_supported():
                raise Finish()

            environment.logger.info('Got new task for synchronizing cluster')
            apply_async(self.synchronize_cluster, environment)
        except Finish:
            pass
        except Exception as e:
            logger.error('Unhandled exception in main Deploy Handler: %s' % e.message)
            self.write_error(500)
        finally:
            self.finish()
