# -*- coding: utf-8 -*-
import json
import re
from collections import defaultdict

from tornado.gen import Return, coroutine, sleep
from tornado.web import Finish

from .base import BaseHandler
from ..core.environment import QloudEnvironment
from ..utils.misc import with_retry, apply_async
from ..utils.logger import logger


class QLOUDHandler(BaseHandler):

    @coroutine
    def post(self):
        try:
            try:
                package = json.loads(self.request.body)
                logger.debug(package)

            except Exception as e:
                message = 'Unable to parse JSON from body => %s' % e.message
                logger.error(message)
                self.write_error(400, message=message)
                raise Finish()

            if 'environmentId' not in package and 'status' not in package and package['environmentId'].count('.') != 2:
                message = 'Malformed JSON body received. Is there is a hook from QLOUD?'
                logger.critical(message)
                self.write_error(400, message=message)
                raise Finish()

            yield self.handle_package(package)

        except Finish:
            pass

        except Exception as e:
            logger.error('Unhandled exception in main QLOUD Handler: %s' % e.message)
            self.write_error(500)

        finally:
            self.finish()

    @coroutine
    def generate_primitives(self, environment):
        dns_zones, qloud_instances, conductor_groups = yield self.fetch_data(environment)
        # now we working w/ conductor hosts and compiled qloud-instance names from qloud
        # containers for syncing w/ conductor and DNS
        _add_to_conductor = list()
        _remove_from_conductor = list()

        _add_to_dns = list()
        _remove_from_dns = list()

        # Searching for new hosts thats are should be added to conductor group
        for fqdn, instance in qloud_instances.items():
            if fqdn in conductor_groups[instance['group']]:
                environment.logger.debug(
                    '%s already in conductor group => %s. Skipping' % (fqdn, instance['group'])
                )
            else:
                conductor_host = {
                    'host': {
                        'fqdn': fqdn,
                        'datacenter_name': instance['root_datacenter_name'].split('-')[0],
                        'description': 'cname for qloud instance %s' % instance['target'],
                        'short_name': fqdn,
                        'group': {
                            'name': instance['group'],
                        }
                    }
                }
                _add_to_conductor.append(conductor_host)
                environment.logger.info(
                    'Missing %s in conductor => %s. Should be added!' % (fqdn, instance['group'])
                )

            # def primitive(self, name, content, type='CNAME', ttl=60, operation='add'):
            # check to what to do with this records = add/remove/update in DNS?
            if fqdn == instance['target'] or not instance['zone']:
                continue

            if ((fqdn + 'CNAME' not in dns_zones[instance['zone']]) and
                    (fqdn + 'AAAA' not in dns_zones[instance['zone']])):
                environment.logger.info('Found missing DNS record for => %s' % fqdn)
                if instance['addresses']:
                    for rtype in ('AAAA', 'A'):
                        target = instance['addresses'][rtype]
                        if target:
                            primitive = self.application.domains_api.primitive(fqdn, target, rtype=rtype)
                            _add_to_dns.append(primitive)
                else:
                    primitive = self.application.domains_api.primitive(fqdn, instance['target'])
                    _add_to_dns.append(primitive)

            else:
                if instance['addresses']:
                    records = (
                        dns_zones[instance['zone']][fqdn + 'AAAA'],
                        dns_zones[instance['zone']].get(fqdn + 'A'),
                    )
                else:
                    records = (dns_zones[instance['zone']][fqdn + 'CNAME'],)
                instance_add_to_dns = dict()
                for record in records:
                    if record is None:
                        continue

                    if ((record['type'] == 'CNAME' and record['right-side'] != instance['target']) or
                            (record['type'] == 'AAAA' and record['right-side'] != instance['addresses']['AAAA']) or
                            (record['type'] == 'A' and record['right-side'] != instance['addresses']['A'])):
                        # remove record first
                        primitive = self.application.domains_api.primitive(
                            fqdn, record['right-side'], rtype=record['type'],
                            ttl=record['ttl'], operation='delete'
                        )
                        _remove_from_dns.append(primitive)

                        if instance['addresses']:
                            for rtype in ('AAAA', 'A'):
                                target = instance['addresses'][rtype]
                                if target:
                                    instance_add_to_dns[rtype] = self.application.domains_api.primitive(
                                        fqdn, target, rtype=rtype
                                    )
                        else:
                            instance_add_to_dns['CNAME'] = self.application.domains_api.primitive(
                                fqdn, instance['target']
                            )

                        environment.logger.info('Found changed DNS record => %s. Recreating it.' % fqdn)
                if instance_add_to_dns:
                    _add_to_dns.extend(instance_add_to_dns.values())

        # Searching for hosts that should be removed from conductor
        for group, hosts in conductor_groups.iteritems():
            for host, info in hosts.iteritems():
                if host not in qloud_instances:
                    environment.logger.info('Found staled host => %s that should be removed from conductor' % host)
                    _remove_from_conductor.append(host)
                    # searching for host in dns_zones

                    _host_in_dns = list()
                    for zone, zone_info in dns_zones.iteritems():
                        for rtype in ('CNAME', 'A', 'AAAA'):
                            host_rtype = host + rtype
                            if host_rtype in zone_info:
                                _host_in_dns.append(zone_info[host_rtype])

                    if not _host_in_dns:
                        environment.logger.debug('Given host => %s not found in DNS. Skipping.' % host)
                        continue

                    for record in _host_in_dns:
                        primitive = self.application.domains_api.primitive(
                            record['left-side'],
                            record['right-side'],
                            rtype=record['type'],
                            ttl=record['ttl'],
                            operation='delete'
                        )
                        _remove_from_dns.append(primitive)
                else:
                    if qloud_instances[host]['group'] != group:
                        environment.logger.info(
                            'There host => %s that should change his group => %s' % (host, info['group'])
                        )
                        _remove_from_conductor.append(host)

                        instance = qloud_instances[host]
                        conductor_host = {
                            'host': {
                                'fqdn': host,
                                'datacenter_name': instance['root_datacenter_name'],
                                'description': 'cname for qloud instance %s' % instance['target'],
                                'short_name': host,
                                'group': {
                                    'name': instance['group']
                                }
                            }
                        }
                        _add_to_conductor.append(conductor_host)
        raise Return((_add_to_conductor, _remove_from_conductor, _add_to_dns, _remove_from_dns))

    @coroutine
    def fetch_data(self, environment):
        cfg = self.application.config['qloud_clusters']
        subdomain = self.application.config['settings']['dns_subdomain_qloud']
        datacenters = self.application.config['settings']['datacenters']
        # containers for important data!
        dns_zones = defaultdict(dict)
        instances = defaultdict(dict)
        conductor_groups = defaultdict(dict)

        env = cfg[environment.project][environment.cluster][environment.environment]
        instance_num_re = re.compile(r'(\d+)$')

        instance_list = yield with_retry(
            self.application.qloud_api.instancelist, (environment,),
            tries=self.application.tries, timeout=self.application.timeout,
        )
        if instance_list is None:
            raise Return()

        plain_instance_list = None
        for instance in instance_list:
            component = instance['componentId'].split('.')[-1]
            if component not in env:
                environment.logger.error('Component => %s not found in env config' % component)
                continue

            if instance['instanceTargetState'] != 'ACTIVE':
                environment.logger.info(
                    'I see instance that did not want to be ACTIVE => %s. Skipping.'
                    % instance['instanceFqdn']
                )
                continue

            component_addresses = None
            if not env[component].get('dns_cname', True):
                if not plain_instance_list:
                    for i in range(self.application.tries):
                        plain_instance_list = yield self.application.qloud_api.plain_instance_list(environment)
                        if plain_instance_list is not None:
                            break
                        yield sleep(self.application.timeout)
                    else:
                        raise Return()

                component_addresses = yield self.application.qloud_api.component_addresses(
                    plain_instance_list, instance['instanceName']
                )

            dns_zone = env[component].get('dns_zone')
            if 'dns_record_format' in env[component] and dns_zone:
                # cast qloud instance format to our internals
                instance_fqdn = env[component]['dns_record_format'].format(
                    num='{0:02d}'.format(int(instance_num_re.search(instance['instanceName']).group(0))),
                    dc=datacenters.get(instance['dc'].lower(), 'u'),
                    subdomain=subdomain,
                    zone=dns_zone,
                )
            else:
                instance_fqdn = instance['instanceFqdn']

            instances[instance_fqdn] = {
                'fqdn': instance_fqdn,
                'target': instance['instanceFqdn'],
                'addresses': component_addresses,
                'group': env[component]['conductor_group'],
                'root_datacenter_name': instance['dc'].lower(),
                'zone': dns_zone
            }

            if dns_zone and dns_zone not in dns_zones:
                # get zone uuid and its records from DNS
                zone_dict = yield self.fetch_dns_zone(environment, subdomain, dns_zone)
                if zone_dict is None:
                    raise Return()
                dns_zones[dns_zone] = zone_dict

            if env[component]['conductor_group'] not in conductor_groups:
                # get hosts from conductor group for given component
                _hostlist = yield self.fetch_conductor(environment, env[component]['conductor_group'])
                if _hostlist is None:
                    raise Return()
                conductor_groups[env[component]['conductor_group']] = _hostlist

        raise Return((dns_zones, instances, conductor_groups))

    @coroutine
    def handle_package(self, package):
        try:
            environment = QloudEnvironment(package, self.application.qloud_api)
            if not environment.is_supported(self.application.config['qloud_clusters']):
                raise Finish()

            if environment.status in self.application.juggler_api.status:
                apply_async(self.application.juggler_api.downtime, environment)

            if environment.status in {'ACTIVATING'}:
                apply_async(self.application.infra_api.notify_deploy, environment)

            # check for deployment status
            if environment.status != 'DEPLOYED' and environment.status != 'REMOVED':
                apply_async(self.application.telegram_api.notify, environment)
                environment.logger.debug('We synchronizing data only when your app in %s state' % 'DEPLOYED')
                raise Finish()

            environment.logger.info('Got new task for synchronizing cluster')
            apply_async(self.synchronize_cluster, environment)
            apply_async(self.application.telegram_api.notify, environment)
            apply_async(self.application.infra_api.resolve_event, environment)
        except Finish:
            pass
        except Exception as e:
            logger.error('Unhandled exception in main QLOUD Handler: %s' % e.message)
            self.write_error(500)
        finally:
            self.finish()
