# -*- coding: utf-8 -*-
import argparse
import logging
import logging.config
import os
import signal
import sys
from multiprocessing.managers import State

import yaml
from tornado import httpserver, ioloop

from .core import Application
from .handlers import QLOUDHandler, TelegramHandler, DeployHandler
from .utils.logger import CONSOLE_LOGGING, logger


def create_app(config, state):
    #: handlers for dispatching our requests
    handlers = [
        (r'/qloud/', QLOUDHandler),
        (r'/deploy/', DeployHandler),
        (r'/telegram/', TelegramHandler),
    ]

    #: application instance
    app = Application(config, handlers, state)
    return app


def run():
    parser = argparse.ArgumentParser(
        description='An API server for syncing clouds w/ conductor.',
        epilog='don`t forget to install yandex-environment-*'
    )
    arguments = parser.add_argument_group('named arguments')
    arguments.add_argument('--config', '-c', type=str, dest='config', required=True, metavar='<config>')

    options = parser.parse_args()

    state = State()
    state.value = State.INITIAL

    if not os.path.exists(options.config) or not os.access(options.config, os.O_RDONLY):
        logger.critical('%s doesn`t exists or not readable. No Config? No Service!' % options.config)
        sys.exit(1)

    try:
        config = yaml.safe_load(open(options.config))
    except Exception as e:
        logger.critical('Unable to load yaml settings, check that config in YAML format: %s' % e.message)
        sys.exit(1)

    app = create_app(config, state)
    server = httpserver.HTTPServer(app, xheaders=True)

    #: ``bind`` used for multi-process purpose // use ``listen`` for single-process mode
    server.bind(config['settings']['bind_port'], address=config['settings']['bind_addr'])

    #: stop signal handler
    def stop(signum, frame):
        if state.value != State.SHUTDOWN:
            state.value = State.SHUTDOWN
            logger.info('Stopping application')

            server.stop()
            ioloop.IOLoop.current().stop()

    #: register signal handling
    signal.signal(signal.SIGINT, stop)
    signal.signal(signal.SIGTERM, stop)

    #: spawn given number of workers or use zero for dynamic spawning
    state.value = State.STARTED
    server.start(config['settings']['workers'])

    logger.info('Application initialized')

    #: start child process activity
    ioloop.IOLoop.instance().start()


def main():
    logging.config.dictConfig(CONSOLE_LOGGING)
    logger.propagate = False
    run()


if __name__ == '__main__':
    main()
