#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
    getip
    ~~~~~~~~~~~~~~~~~~~~~~~

    Parse given netinfo macro and other params from racktables
    get ip-info from racktables w/ bisection algorithm. returns first free ip
"""
import argparse
import logging.config
import sys

from getip.main import get_ip
from getip.settings import DC_MAP
from getip.utils.logger import CONSOLE_LOGGING, logger


def main():
    parser = argparse.ArgumentParser(description='Utility for searching free IP addresses in a given Macro/VLAN/DC')

    #: two groups of parameters => flags and named variables
    flags = parser.add_argument_group('Boolean flags arguments')
    flags.add_argument(
        '--update', '-u', dest='update', help='Download fresh content from RT',
        action='store_true', default=False, required=False
    )
    flags.add_argument(
        '--debug', '-d', dest='debug', help='Enable debugging',
        action='store_true', default=True, required=False
    )
    flags.add_argument(
        '--v6', '-6', dest='v6', help='Allocate V6 address',
        action='store_true', default=False, required=False
    )
    flags.add_argument(
        '--v4', '-4', dest='v4', help='Allocate V4 address',
        action='store_true', default=False, required=False
    )
    flags.add_argument(
        '--quiet', '-q', dest='quiet', help='Be quiet',
        action='store_true', default=False, required=False
    )
    flags.add_argument(
        '--many', '-m', dest='many', help='Allocate many addresses',
        type=int, default=1, required=False
    )

    arguments = parser.add_argument_group('Named arguments')
    arguments.add_argument(
        '--macro', help='Network macros, e.g. _DISKNETS_',
        type=str, dest='macro', required=True, metavar='<macros name>'
    )
    arguments.add_argument(
        '--vlan', help='VLAN ID. No description here.',
        type=str, dest='vlan', required=True, metavar='<vlan id>'
    )
    arguments.add_argument(
        '--dc', help='DC Name. One of: {}'.format(', '.join(DC_MAP.keys())),
        choices=DC_MAP.keys(), type=str, dest='dc', required=True, metavar='<dc>'
    )

    #: cuz we have required arguments => we print help if sys.argv is empty
    if len(sys.argv) == 1:
        parser.print_help()
        sys.exit(1)

    options = parser.parse_args()

    if not options.quiet:
        logging.config.dictConfig(CONSOLE_LOGGING)

    if not options.v6 and not options.v4:
        logger.critical('You must provide IP version w/ options <white>--v6</white> or/with <white>--v4</white>')
        sys.exit(1)

    found_ips = get_ip(options)
    if options.v4:
        print(" ".join(found_ips[4]))
    if options.v6:
        print(" ".join(found_ips[6]))


if __name__ == '__main__':
    main()
