# -*- coding: utf-8 -*-
import os
import random
import socket
import sys
import time
from collections import namedtuple

import netaddr
import requests
import requests.exceptions
from getip import settings
from getip.utils.logger import logger

socket.setdefaulttimeout(5)


def get_networks(options):
    networks = list()
    if options.update and os.path.exists(settings.CACHED_MACROS_FILE):
        logger.debug('Invalidating cache as requested')
        os.remove(settings.CACHED_MACROS_FILE)

    #: trying to get cache
    if not os.path.exists(settings.CACHED_MACROS_FILE):
        res = requests.get(settings.EXPORT_NETS_URL)

        if not 200 <= res.status_code <= 300:
            logger.error(
                'Failed to request <white>{URL}</white> '
                'code => <red>{code}</red> reason => <yellow>{reason}</yellow>'.format(
                    URL=settings.EXPORT_NETS_URL, code=res.status_code, reason=res.reason
                )
            )
            sys.exit(1)

        content = res.content
        with open(settings.CACHED_MACROS_FILE, 'w') as cached_macros_file:
            cached_macros_file.write(content)

    else:
        content = open(settings.CACHED_MACROS_FILE).read()

    model = namedtuple('Network', ['CIDR', 'Macro', 'VLAN', 'Project', 'Responsibles', 'DC'])
    #: positive answer! continue
    for row in content.split('\n')[1:]:
        row = row.split('\t')

        if len(row) != len(model._fields):
            #: strange length of parameters // skip this row
            continue

        netinfo = model._make(row)
        if (netinfo.Macro != options.macro or
                netinfo.VLAN != options.vlan or
                netinfo.DC.decode('utf-8') not in settings.DC_MAP[options.dc]):
            continue

        addr, cidr = netinfo.CIDR.split('/')

        #: skip unnecessary ranges
        if not options.v4 and netaddr.valid_ipv4(addr):
            logger.debug('Skipping IPv4 netinfo => <gray>{}</gray>'.format(netinfo.CIDR))
            continue

        if not options.v6 and netaddr.valid_ipv6(addr):
            logger.debug('Skipping IPv6 netinfo => <gray>{}</gray>'.format(netinfo.CIDR))
            continue

        logger.info(
            'Found netinfo => <gray>{netinfo}</gray> datacenter => <yellow>{dc}</yellow>'
            .format(netinfo=netinfo.CIDR, dc=netinfo.DC)
        )
        networks.append(netinfo)
    return networks


def get_ip(options):
    try:
        found_ips = {4: [], 6: []}
        #: allocate one IP per netinfo
        for netinfo in get_networks(options):
            network = netaddr.IPNetwork(netinfo.CIDR)

            #: choose maximum border
            length = sys.maxint if network.size > sys.maxint else len(network) - 1
            started = time.time()

            for _ in range(options.many):
                while time.time() - started <= settings.PROBING_TIMEOUT:
                    idx = random.randrange(1, length)
                    random_ip = str(network[idx])

                    #: probing network
                    logger.debug('Probing DNS and switch info for IP => <gray>{}</gray>'.format(random_ip))
                    try:
                        if socket.gethostbyaddr(random_ip):
                            logger.debug('This ip already have reverse DNS record. Skipping...')
                            continue

                    except socket.herror:
                        #: Ok, this IP doesn't have reverse zone. but it can be assigned to someone HW
                        try:
                            res = requests.get(settings.SWITCH_INFO_URL.format(ip=random_ip))
                            if 200 <= res.status_code < 300:
                                if 'switch' in res.content:
                                    logger.debug('This IP doesnt have PTR but assigned to switchport')
                                    continue
                                logger.debug('Addr is not assigned anywhere => <gray>{}</gray>'.format(random_ip))
                            else:
                                logger.error('Some error white getting switch info => {}'.format(res.content))
                                continue
                        except requests.exceptions.BaseHTTPError as e:
                            logger.debug('Failed to get switchinfo for this ip => <gray>{}</gray>'.format(random_ip))
                            continue
                    logger.info('Found unassigned IP => <yellow>{}</yellow>'.format(random_ip))
                    found_ips[network.version].append(random_ip)
                    break
        return found_ips

    except Exception as e:
        logger.exception('Unhandled exception: %s' % e)
