# -*- coding: utf-8 -*-
"""
    pkginfo.utils.logging
    ~~~~~~~~~~~~~~~~~

    Logging settings which are using colored logs package.
"""
import logging.config
from functools import wraps

CONSOLE_COLOR_MAP = {
    'DEBUG': 'blue',
    'INFO': 'green',
    'WARNING': 'bold_yellow',
    'ERROR': 'red',
    'CRITICAL': 'bold_red'
}

RESET_CODE = '\033[0m'

OPENING_COLORS_TAGS = dict(
    zip(
        [
            '<gray>',
            '<red>',
            '<green>',
            '<yellow>',
            '<blue>',
            '<magenta>',
            '<cyan>',
            '<white>',
        ],
        map(lambda x: '\033[%dm' % x, range(30, 38))
    )
)
OPENING_COLORS_TAGS['<bold_red>'] = '\033[1;31m'
OPENING_COLORS_TAGS['<bold_yellow>'] = '\033[1;33m'

CLOSING_COLORS_TAGS = dict(
    zip(
        [
            '</gray>',
            '</red>',
            '</green>',
            '</yellow>',
            '</blue>',
            '</magenta>',
            '</cyan>',
            '</white>',
            '</bold_red>',
            '</bold_yellow>'
        ],
        [RESET_CODE] * len(OPENING_COLORS_TAGS)
    )
)

CONSOLE_COLOR_FORMAT = '%(asctime)s %(black)s%(levelname)-8s %(reset)s %(log_color)s%(message)s'

CONSOLE_LOGGING = {
    'version': 1,
    'formatters': {
        'colored': {
            '()': 'colorlog.ColoredFormatter',
            'format': CONSOLE_COLOR_FORMAT,
            'datefmt': '%H:%M:%S',
            'reset': True,
            'log_colors': CONSOLE_COLOR_MAP
        }
    },
    'handlers': {
        'default': {
            'formatter': 'colored',
            'level': 'DEBUG',
            'class': 'logging.StreamHandler',
        },
    },
    'loggers': {
        'disk-getip': {
            'handlers': ['default'],
            'level': 'DEBUG',
            'propagate': True
        }
    }
}


def ColoredCall(fn, method):
    @wraps(fn)
    def wrapped(message, *args, **kwargs):
        default_color = OPENING_COLORS_TAGS.get('<{color}>'.format(
            color=CONSOLE_COLOR_MAP.get(method.upper()))
        )
        NEW_COLOR = '{RESET_CODE}{NEW_COLOR}'.format(
            RESET_CODE=RESET_CODE, NEW_COLOR=default_color
        )

        for pack in [OPENING_COLORS_TAGS, CLOSING_COLORS_TAGS]:
            for tag, code in pack.iteritems():
                message = message.replace(tag, code)

            message = message.replace(RESET_CODE, NEW_COLOR)

        fn(message, *args, **kwargs)

    return wrapped


def ColoredLogger(logger):
    for method in map(str.lower, CONSOLE_COLOR_MAP):

        call = getattr(logger, method)
        call = ColoredCall(call, method)

        setattr(logger, method, call)

    return logger


#: Note, you must `logging.config.dictConfig(CONSOLE_LOGGING)` manually
logger = logging.getLogger('disk-getip')
logger = ColoredLogger(logger)
