#!/usr/bin/env python

# -*- coding: UTF-8 -*-

import re
import sys
from collections import defaultdict, Counter

request_re = re.compile('https?:\/\/(?P<b_host>[^\/:]+)')
response_re = re.compile('\((?P<b_request_time>\d+)ms\)')
nodejs_parseable_error_re = re.compile('^(?:\[console|down\]|\[\d+\.\d+\])')
nodejs_error_re = re.compile('^\[console')
nodejs_exception_re = re.compile('\]\s+(?P<b_exception_type>[a-zA-Z]+?Error):')
nodejs_down_re = re.compile('^\[down')
nodejs_block_error_re = re.compile('\[\d+\.\d+\]\s\[(?P<b_error_block_type>[^\s]+)')

results_request_timings = defaultdict(Counter)
results_response_timings = Counter()

results_request_count_codes = defaultdict(Counter)
results_request_aggr_count_codes = defaultdict(Counter)

results_nodejs_count_errors = Counter()
results_nodejs_count_exceptions = Counter()

results_nodejs_aggr_count_errors = Counter()
results_nodejs_aggr_count_exceptions = Counter()
results_nodejs_aggr_count_downs = Counter()


def pack_timings(result_timings):
    return map(lambda t: "%s@%s" % t, sorted(result_timings.items()))


for line in sys.stdin:
    if line.strip() == '':
        continue

    parts = line.strip("\n").split("\t")
    parsed = {}
    for part in parts:
        eq = part.find('=')
        key = part[:eq]
        value = part[(eq + 1):]
        parsed[key] = value
    level = parsed['level']
    message = parsed['message']

    if level == 'error':
        matches = nodejs_parseable_error_re.search(message)
        if matches:
            error_match = nodejs_error_re.search(message)
            down_match = nodejs_down_re.search(message)
            block_error_match = nodejs_block_error_re.search(message)
            if error_match:
                exception_match = nodejs_exception_re.search(message)
                if exception_match:
                    results_nodejs_count_exceptions[exception_match.group('b_exception_type').lower()] += 1
                else:
                    results_nodejs_count_errors['other'] += 1
            elif down_match:
                results_nodejs_aggr_count_downs['total'] += 1
            elif block_error_match:
                results_nodejs_count_errors[block_error_match.group('b_error_block_type')] += 1

    if 'time' not in parsed:
        parsed['time'] = '0'

    if '[request' in message:
        results_response_timings[parsed['time']] += 1

    if '[http' in message and 'http_url' in parsed:
        matches = request_re.search(parsed['http_url'])
        if matches:
            u_host = matches.group('b_host').replace('.', '_')
            u_status = int(parsed['http_status'])

            results_request_timings[u_host][parsed['time']] += 1
            results_request_count_codes[u_host][u_status] += 1


# aggregate
for u_host, result in sorted(results_request_count_codes.iteritems()):

    for u_status_k, u_status_v in sorted(result.iteritems()):
        acode = 507 if u_status_k == 507 else str(u_status_k / 100) + 'xx'

        results_request_aggr_count_codes[u_host][acode] += u_status_v
        results_request_aggr_count_codes[u_host]['total'] += u_status_v
        results_request_aggr_count_codes['total']['total'] += u_status_v

for error_k, error_v in sorted(results_nodejs_count_errors.iteritems()):
    results_nodejs_aggr_count_errors['total'] += error_v

for exception_k, exception_v in sorted(results_nodejs_count_exceptions.iteritems()):
    results_nodejs_aggr_count_exceptions['total'] += exception_v

# request count codes
for u_host, result in sorted(results_request_count_codes.iteritems()):
    for u_status_k, u_status_v in sorted(result.iteritems()):
        print("request_count_code_{}_{} {}".format(u_host, u_status_k, u_status_v))

# request count aggr codes
for u_host, result in sorted(results_request_aggr_count_codes.iteritems()):
    for u_status_k, u_status_v in sorted(result.iteritems()):
        print("request_aggr_count_code_{}_{} {}".format(u_host, u_status_k, u_status_v))

# request timings
for u_host, timings in sorted(results_request_timings.iteritems()):
    if len(timings):
            print("@request_timings_{} {}".format(u_host, ' '.join(pack_timings(timings))))

# response timings
print("@response_timings {}".format(' '.join(pack_timings(results_response_timings))))

# errors and exceptions
for error_k, error_v in sorted(results_nodejs_count_errors.iteritems()):
    print("nodejs_count_error_{} {}".format(error_k, error_v))

for exception_k, exception_v in sorted(results_nodejs_count_exceptions.iteritems()):
    print("nodejs_count_exception_{} {}".format(exception_k, exception_v))

for error_k, error_v in sorted(results_nodejs_aggr_count_errors.iteritems()):
    print("nodejs_aggr_count_error_{} {}".format(error_k, error_v))

for exception_k, exception_v in sorted(results_nodejs_aggr_count_exceptions.iteritems()):
    print("nodejs_aggr_count_exception_{} {}".format(exception_k, exception_v))

for down_k, down_v in sorted(results_nodejs_aggr_count_downs.iteritems()):
    print("nodejs_aggr_count_down_{} {}".format(down_k, down_v))
