#!/bin/bash
JDK_VERSION=1.8
set -e

SAVED="`pwd`"
cd "$(dirname `dirname "${BASH_SOURCE}[0]"`)"
APP_DIR="`pwd -P`"
cd ${SAVED} >> /dev/null

USER=logshatter
SERVICE_NAME=logshatter

if [[ -d "$APP_DIR/jdk"  ||  -L "$APP_DIR/jdk" ]] ; then
    JAVA_HOME=$APP_DIR/jdk
else
    JAVA_HOME=/usr/local/java8
fi

PROPERTIES_DIR=${APP_DIR}/properties.d
BIN_DIR=${APP_DIR}/bin

PORT_PARAMETERS=""
DEBUG_PORT=""
JMX_PORT=""

function get_fqdn () {
    host_fqdn=$(/bin/hostname -f)
    if [ -z $host_fqdn ]; then
        host_fqdn=$(/bin/uname -n)
    fi
    if [ -z $host_fqdn ]; then
        (>&2 echo "Failed to get fqdn")
        exit 1
    fi
    echo $host_fqdn
}

HOST_NAME=$(/bin/hostname -s)
HOST_FQDN=$(get_fqdn)

function add_jvm_port_param () {
    PORT_PARAM=`python -c "import re, sys; matches = re.match('(.*port)(=\\d+)$', sys.argv[1].lower()); print '-D' + re.sub('\.?port$', '.port', re.sub('[^a-z0-9]+', '.', matches.group(1))) + matches.group(2) if matches else ''" $1`
    if [ ! -z "${PORT_PARAM}" ] ; then
        PORT_PARAMETERS="${PORT_PARAMETERS} ${PORT_PARAM}"
        D_PORT=${PORT_PARAM/'-Ddebug.port='/''}
        if [ ${D_PORT} != ${PORT_PARAM} ] ; then DEBUG_PORT=${D_PORT} ; fi

        J_PORT=${PORT_PARAM/'-Djmx.port='/''}
        if [ ${J_PORT} != ${PORT_PARAM} ] ; then JMX_PORT=${J_PORT} ; fi
    fi
}

function get_cpu_count () {
    cpu_count_default=1
    # check if it's Runtime Cloud instance
    if [ -e dump.json ]; then
        cpu_count=`python -c 'import json; import os; data=json.load(open("dump.json", "r")); cpu_guarantee=data["container"]["constraints"]["cpu_guarantee"].strip("c") if "cpu_guarantee" in data["container"]["constraints"] else os.environ["CPU_GUARANTEE"].strip("c"); print(1 if float(cpu_guarantee) < 1 else int(float(cpu_guarantee)))' 2>/dev/null || echo $cpu_count_default`
    else
        cpu_count=`grep -c processor /proc/cpuinfo 2>/dev/null || echo $cpu_count_default`
    fi
    echo $cpu_count
}

#Read environment from file
ENV_TYPE_FILE=/etc/yandex/environment.type
if [ -e ${ENV_TYPE_FILE} ] ; then
    ENVIRONMENT=$(cat ${ENV_TYPE_FILE})
else
    ENVIRONMENT=development
fi

#Read properties from script keys
for opt in "$@"; do
  case ${opt} in
    --environment=*) ENVIRONMENT="${opt#*=}"
    shift ;;
    --cpu-count=*) CPU_COUNT="${opt#*=}"
    shift ;;
    --logdir=*) LOG_DIR="${opt#*=}"
    shift ;;
    --tmpdir=*) TMP_DIR="${opt#*=}"
    shift ;;
    --datadir=*) DATA_DIR="${opt#*=}"
    shift ;;
    --extdatadir=*) EXT_DATA_DIR="${opt#*=}"
    shift ;;
    --hostname=*) HOST_NAME="${opt#*=}"
    shift ;;
    --hostname-fqdn=*) HOST_FQDN="${opt#*=}"
    shift ;;
    --propertiesdir=*) PROPERTIES_DIR="${opt#*=}"
    shift ;;
    --*port=*) add_jvm_port_param "${opt:2}"
    shift ;;
    --java-home=*) JAVA_HOME="${opt#*=}"
    shift ;;
    --dc=*) SRV_DC="${opt#*=}"
    shift ;;
    *)

    ;;
  esac
done

#Find values in *.properties of application
function find_in_path () { if [ -d "${2}" ]; then VAL=`grep "^${1}=" "${2}"/*.properties | head -n 1`; echo ${VAL#*=} ; fi }
function find_in_properties () {
    if [ ! -z "$1" ]
        then echo "$1"
    else
        VAL=$(find_in_path "${2}" ${PROPERTIES_DIR}/${ENVIRONMENT})
        if [ -z "${VAL}" ] ; then
            find_in_path "${2}" ${PROPERTIES_DIR};
        else
            echo "${VAL}"
        fi
    fi
};

LOG_DIR=$(find_in_properties "${LOG_DIR}" "log.dir")
TMP_DIR=$(find_in_properties "${TMP_DIR}" "tmp.dir")
DATA_DIR=$(find_in_properties "${DATA_DIR}" "data.dir")
EXT_DATA_DIR=$(find_in_properties "${EXT_DATA_DIR}" "ext.data.dir")
DEBUG=$(find_in_properties "${DEBUG}" "debug")
CPU_COUNT=$(find_in_properties "${CPU_COUNT}" "cpu.count")
TMP_DIR=$(find_in_properties "${TMP_DIR}" "tmp.dir")
GC_LOG_FILE_CNT=$(find_in_properties "${GC_LOG_FILE_CNT}" "gc.log.file_cnt")
GC_LOG_FILE_SIZE=$(find_in_properties "${GC_LOG_FILE_SIZE}" "gc.log.file_max_size")

if [ -z "${DEBUG_PORT}" ] ; then DEBUG_PORT=$(find_in_properties "${DEBUG_PORT}" "debug.port") ; fi
if [ -z "${LOG_DIR}" ] ; then LOG_DIR="${APP_DIR}/log" && test -d ${LOG_DIR} || mkdir ${LOG_DIR}; fi
if [ -z "${DATA_DIR}" ] ; then DATA_DIR="${APP_DIR}/data" && test -d ${DATA_DIR} || mkdir ${DATA_DIR}; fi
if [ -z "${CPU_COUNT}" ] ; then CPU_COUNT=$(get_cpu_count) ; fi
if [ -z "${JMX_PORT}" ] ; then JMX_PORT=$(find_in_properties "${JMX_PORT}" "jmx.port") ; fi
if [ -z "${GC_LOG_FILE_CNT}" ] ; then GC_LOG_FILE_CNT="3" ; fi
if [ -z "${GC_LOG_FILE_SIZE}" ] ; then GC_LOG_FILE_SIZE="30M" ; fi

if [ -z "" ] ; then
    HEAP_DUMP_PATH=${LOG_DIR}/hprof/
    mkdir -p ${HEAP_DUMP_PATH}
else
    HEAP_DUMP_PATH="${LOG_DIR}/"
    mkdir -p $(dirname "${HEAP_DUMP_PATH}")
fi

LOG_GC="${LOG_DIR}/${SERVICE_NAME}.gc.log"
if [ -e $LOG_GC ] ; then
    mv $LOG_GC $LOG_GC.1
fi
if [ -e "${LOG_GC}.0.current" ] ; then
    mv "${LOG_GC}.0.current" "${LOG_GC}.0.old"
fi

GC_OPTIONS="-XX:+PerfDisableSharedMem"
if [[ ${JDK_VERSION} == "1.8"  ]] ; then
    GC_LOG_OPT="-Xloggc:$LOG_GC"
else
    GC_OPTIONS="-XX:+AlwaysActAsServerClassMachine ${GC_OPTIONS}"
    GC_LOG_OPT="-Xlog:gc*:file=$LOG_GC:time,uptime:filecount=${GC_LOG_FILE_CNT},filesize=${GC_LOG_FILE_SIZE}"
fi

if [ ! -z "${DEBUG}" ]; then DEBUG=${DEBUG/'$DEBUG_PORT'/${DEBUG_PORT}}; fi

#empty mean default(configured from app)
LOG4J2_CONFIG=""
if [ -s "${APP_DIR}/conf/log4j2.xml" ]; then
    LOG4J2_CONFIG="${APP_DIR}/conf/log4j2.xml"
fi
if [ -s "${APP_DIR}/conf/log4j2-override.xml" ]; then
    if [ ! -z "${LOG4J2_CONFIG}" ] ; then
        LOG4J2_CONFIG="${LOG4J2_CONFIG},${APP_DIR}/conf/log4j2-override.xml"
    else
        LOG4J2_CONFIG="${APP_DIR}/conf/log4j2-override.xml"
    fi
fi

JAVA_OPTS="${DEBUG} -Xss2m -Djava.net.preferIPv6Addresses=true -Djava.net.preferIPv4Stack=false -XX:+HeapDumpOnOutOfMemoryError -XX:HeapDumpPath=${LOG_DIR}/hprof/ -showversion -server -verbose:gc -XX:+PrintGC -XX:+PrintGCDetails -XX:+PrintGCTimeStamps -XX:+PrintGCDateStamps -Xloggc:$LOG_GC -XX:+UseCompressedOops -XX:AutoBoxCacheMax=10000 -Dbean.file=classpath:logshatter.xml -Ddc=${SRV_DC,,} -Dmodule.properties.file=70_logshatter.properties -Dfile.encoding=UTF-8"
JAVA_OPTS="${JAVA_OPTS} -XX:HeapDumpPath=${HEAP_DUMP_PATH} -XX:+HeapDumpOnOutOfMemoryError -XX:OnOutOfMemoryError=\"chmod go+r ${HEAP_DUMP_PATH}*\" ${GC_OPTIONS} ${GC_LOG_OPT} -Dhost.name=${HOST_NAME} -Dhost.fqdn=${HOST_FQDN} -Dcpu.count=${CPU_COUNT} "
JAVA_OPTS="${JAVA_OPTS} -Dlog.dir=${LOG_DIR} -Dtmp.dir=${TMP_DIR} -Ddata.dir=${DATA_DIR} -Dext.data.dir=${EXT_DATA_DIR} -Denvironment=${ENVIRONMENT} -Dconfigs.path=${PROPERTIES_DIR} -Dapp.dir=${APP_DIR}"
JAVA_OPTS="${JAVA_OPTS} ${PORT_PARAMETERS}"
JAVA_OPTS="${JAVA_OPTS} -Dlog4j2.formatMsgNoLookups=true -Dlog4j.formatMsgNoLookups=true"

if [ ! -z "${LOG4J2_CONFIG}" ]; then JAVA_OPTS="${JAVA_OPTS} -Dlog4j.configurationFile=${LOG4J2_CONFIG}" ; fi

#For jni libraries
LD_LIBRARY_PATH="${APP_DIR}/lib:$LD_LIBRARY_PATH"

#For binary files
PATH="${APP_DIR}/bin:$PATH"

#Deprecated
JAVA_OPTS="${JAVA_OPTS} -DLOG_DIR=${LOG_DIR}"

ENV_JVM_ARGS=$(find_in_properties "${ENV_JVM_ARGS}" "environmentJvmArgs")
if [ ! -z "${ENV_JVM_ARGS}" ]; then JAVA_OPTS="${JAVA_OPTS} ${ENV_JVM_ARGS}" ; fi

if [ ! -z "${TMP_DIR}" ] && [ -d "${TMP_DIR}" ] ; then JAVA_OPTS="${JAVA_OPTS} -Djava.io.tmpdir=${TMP_DIR}" ; fi

export JAVA_HOME JAVA_OPTS LOG_DIR ENVIRONMENT PROPERTIES_DIR LD_LIBRARY_PATH PATH

PID_FILE=/var/run/$USER/$SERVICE_NAME.pid
NOW=`date`
echo "[${NOW}]: start $BIN_DIR/${SERVICE_NAME} with parameters ${JAVA_OPTS}"
. "$BIN_DIR/${SERVICE_NAME}"
