#!/usr/bin/env python
# -*- coding: UTF-8 -*-
import re
import sys
from collections import Counter, defaultdict


def tskv_get_key(input_line, input_key):
    key_index = input_line.find('\t%s=' % input_key)
    return input_line[key_index + (len(input_key) + 2):].split('\t')[0]


def json_get_key(obj, input_key):
    key_index = obj.find('\\"%s\\":' % input_key)
    return obj[key_index + (len(input_key) + 5) + 2:].split('\\')[0]


def metric_name(*args):
    return '_'.join(map(str, args))


def print_codes(name, counter):
    for k, v in counter.items():
        print("%s.%s %d" % (name, k, v))


def print_timings(name, input_timings):
    packed_timings = map(lambda t: "%s@%s" % t, sorted(input_timings.items()))
    if packed_timings:
        print("@%s %s" % (name, ' '.join(packed_timings)))


results_count_codes = Counter()
results_timings = Counter()
protocol_count_codes = Counter()
protocol_timings = defaultdict(Counter)

handlers_count_codes = Counter()
handlers_timings = defaultdict(Counter)

anonymous_group = "anonymous"
inplace_group = "inplace"

end_of_url = "([/]?$|[/]?\?)"

def compile(url):
    return re.compile(url + end_of_url)

def url_block_by_group(group_name):
    return "/(?P<%s>[^ /]+)" % group_name


def url_block(inplace, name):
    return url_block_by_group("%s_%s" % (inplace_group, name)) if inplace \
        else url_block_by_group("%s_%s" % (anonymous_group, name))


product_set = url_block(True, "product_set")
group_type = url_block(True, "groupType")
group_id = url_block(False, "group_id")
product_id = url_block(False, "product_id")
request_id = url_block(False, "request_id")
uid = url_block(False, "uid")
order_id = url_block(False, "order_id")
service_id = url_block(False, "service_id")
feature_code = url_block(False, "feature_code")
binding_id = url_block(False, "binding_id")
card_external_id = url_block(False, "card_external_id")
promo_code = url_block(False, "promo_code")
promo_key = url_block(False, "promoKey")

# порядок важен! считаем первый сматчившийся регекс нужным
regexes = [compile("/v1/directory/userUpdated"),
           compile("/v1/mailsupport/groups" + group_type + group_id + "/services"),
           compile("/v1/mailsupport/users/services"),
           compile("/v1/mailsupport/group/make_free_subscription"),
           compile("/v1/mailsupport/group/revoke_subscription"),
           compile("/v1/groups/products" + product_id),
           compile("/v1/groups/promo_code/activate"),
           compile("/v1/groups/promo_code/check"),
           compile("/v1/groups/promo_code" + promo_code + "/activate"),
           compile("/v1/groups/promo_code" + promo_code + "/check"),
           compile("/v1/groups" + group_type + group_id + "/services"),
           compile("/v1/groups" + group_type + group_id + "/addons"),
           compile("/v1/groups" + group_type + group_id + "/reset_payment_data"),
           compile("/v1/groups" + group_type + group_id + "/accept_agreement"),
           compile("/v1/groups" + group_type + group_id + "/invoices"),
           compile("/v1/groups" + group_type + group_id + "/payment/trust-form"),
           compile("/v2/groups" + group_type + group_id + "/payment/trust-form"),
           compile("/v1/groups" + group_type + group_id + "/payment/cards/binding_form"),
           compile("/v1/groups" + group_type + group_id + "/feature" + feature_code),
           compile("/v1/groups" + group_type + group_id + "/payment/cards"),
           compile("/v1/groups" + group_type + group_id + "/payment/cards" + card_external_id),
           compile("/v1/groups" + group_type + group_id + "/payment/settings/autobilling"),
           compile("/v1/groups" + group_type + "/filter_by_accepted_agreement"),
           compile("/v1/groups/organization" + group_id + "/subscribe_with_payment_data"),
           compile("/v1/groups/organization" + group_id + "/ph/subscribe_with_payment_data"),
           compile("/v1/groups/organization" + group_id + "/subscribe"),
           compile("/v1/groups/organization" + group_id + "/subscribe_for_addon"),
           compile("/v1/groups/organization" + group_id + "/unsubscribe"),
           compile("/v1/groups/organization" + group_id + "/payment_data"),
           compile("/v1/groups/organization" + group_id + "/ph/payment_data"),
           compile("/v1/groups/organization/by_payer"),
           compile("/v1/groups/organization/available_payment_data"),
           compile("/v2/groups/organization/available_payment_data"),
           compile("/v1/groups/organization/admins" + uid + "/features"),
           compile("/v1/groups/transactions/recalculation"),
           compile("/v1/groups/productsets" + product_set + "/products"),
           compile("/v1/groups" + group_type + group_id),
           compile("/v1/payments/request" + request_id + "/notify"),
           compile("/v1/promo/future"),
           compile("/v1/promo" + product_set + "/activate"),
           compile("/v1/users/orders/refresh"),
           compile("/v1/users/subscribe"),
           compile("/v1/users/unsubscribe"),
           compile("/v1/users/services"),
           compile("/v1/orders" + order_id + "/notify"),
           compile("/v1/users/inapp/payment_initialized"),
           compile("/v1/users/inapp/receipt"),
           compile("/v2/users/inapp/receipt"),
           compile("/v1/orders" + order_id),
           compile("/v1/users/services" + service_id + "/refund"),
           compile("/v1/users/passport/subscriptions_statuses"),
           compile("/v1/productsets" + product_set + "/products"),
           compile("/v2/productsets" + product_set + "/products"),
           compile("/v3/productsets" + product_set + "/products"),
           compile("/v1/payments/binding" + binding_id + "/notify"),
           compile("/v1/promocode/codes" + promo_code + "/activate"),
           compile("/v1/admin/promo" + promo_key + "/payload"),
           compile("/v1/promocode/activate"),
           compile("/v1/promocode/generate")
           ]

lines = sys.stdin
for line in lines:
    if not line.strip():
        continue

    u_status = tskv_get_key(line, 'status')
    a_status = u_status[0] + 'xx'
    request_time = tskv_get_key(line, 'request_time')

    results_count_codes[u_status] += 1
    results_count_codes[a_status] += 1
    results_count_codes['total'] += 1
    results_timings[request_time] += 1

    protocol = tskv_get_key(line, 'protocol').replace('/', '_')
    protocol_count_codes[metric_name(protocol, u_status)] += 1
    protocol_count_codes[metric_name(protocol, a_status)] += 1
    protocol_timings[protocol][request_time] += 1

    request = tskv_get_key(line, 'request')
    regex = next((regex for regex in regexes if regex.match(request)), None)
    if regex:
        handler = regex.pattern
        match = regex.match(request)

        for group_name in regex.groupindex:
            if group_name.startswith(inplace_group):
                groupIndex = regex.groupindex.get(group_name) - 1
                handler = handler.replace(url_block_by_group(group_name), "/" + match.groups()[groupIndex])

            if group_name.startswith(anonymous_group):
                groupIndex = regex.groupindex.get(group_name) - 1
                handler = handler.replace(url_block_by_group(group_name),
                                          "/" + group_name.replace("%s_" % anonymous_group, ""))

        handler = handler.strip('/').replace(end_of_url, '').replace('/', '.')
        handlers_count_codes[metric_name(handler, u_status)] += 1
        handlers_timings[handler][request_time] += 1

print_codes("nginx_count_request_total_code", results_count_codes)
print_timings("nginx_timings_total", results_timings)

print_codes("protocol_count_codes", protocol_count_codes)
for key, timings in protocol_timings.items():
    print_timings("protocol_timings.%s" % key, timings)

print_codes("handlers_count_codes", handlers_count_codes)
for key, timings in handlers_timings.items():
    print_timings("handlers_timings.%s" % key, timings)
