#!/usr/bin/env python2.7
# -*- coding: utf-8 -*-
import sys
from datetime import datetime, timedelta

import OpenSSL
import requests
import yaml
from startrek_client import Startrek
from vault_client.instances import Production

requests.packages.urllib3.disable_warnings()


def get_group_members(token, department):
    response = requests.get(
        'https://staff-api.yandex-team.ru/v3/persons',
        params={
            'department_group.url': department,
            'official.is_dismissed': 'false',
            '_fields': 'login'
        },
        headers={'Authorization': 'OAuth %s' % token},
        verify=False,
    )
    return {u['login'] for u in response.json()['result']}


class StartrekClient(Startrek):
    disk_user_agent = 'Mozilla/5.0 (Windows NT 5.1; rv:31.0) Gecko/20100101 Firefox/31.0'

    def __init__(self, token, **kwargs):
        # Get token here:
        # https://oauth.yandex-team.ru/authorize?response_type=token&client_id=a7597fe0fbbf4cd896e64f795d532ad2
        super(StartrekClient, self).__init__(useragent=self.disk_user_agent, token=token, **kwargs)
        self._connection.session.verify = False

    def get_issues(self, query):
        """Получение тикетов"""
        return list(self.issues.find(filter=query))

    def create_issue(self, **kwargs):
        return self.issues.create(**kwargs)


class TicketError(Exception):
    pass


class NotExistsError(TicketError):
    pass


class ErrorTicket(object):
    """
    Обычный тикет об ошибке
    """
    follower_logins = set()
    is_uniq = True
    issue_dict = {
        'queue': 'CHEMODAN',
        'type': {'name': 'Ошибка'},
    }

    def __init__(self, summary, assignee=None, followers=[], queue='CHEMODAN', token=None):
        self.st_client = StartrekClient(token)
        self.summary = summary
        self.search_query = {
            'queue': queue,
            'summary': summary,
        }
        self.assignee_login = assignee
        self._issues = None
        self._issues_in_release = None
        self.issue_dict = {
            'queue': queue,
            'type': {'name': 'Ошибка'},
            'followers': [{'login': l} for l in (self.follower_logins | set(followers))],
        }

    @property
    def issues(self):
        """
        Получить тикеты, попадающие под `search_query`

        Проверяет тикет на соответсвие флагу `is_uniq`
        """
        # возвращает список
        if not self._issues:
            self._issues = self.st_client.get_issues(self.search_query)
            num = len(self._issues)
            if num == 0:
                raise NotExistsError()
        return self._issues

    def is_can_be_created(self):
        # если тикет не должен быть уникальным, то ок
        if not self.is_uniq:
            return True

        try:
            self.issues
        except NotExistsError:
            # не нашли тикет - ok
            return True
        else:
            return False

    def create(self, description='', components=[], deadline=None):
        """Создание тикета"""
        if not isinstance(description, unicode):
            raise TypeError('Bad description')

        if self.is_can_be_created():
            issue_dict = self.issue_dict.copy()
            issue_dict['summary'] = self.summary
            issue_dict['description'] = description
            if components:
                issue_dict['components'] = components
            if deadline:
                issue_dict['deadline'] = deadline
            if self.assignee_login:
                issue_dict['assignee'] = {'login': self.assignee_login}
            issue = self.st_client.create_issue(**issue_dict)
            print "CREATED https://st.yandex-team.ru/%s" % issue.key
        else:
            for iss in self._issues:
                if self.search_query['summary'] == iss.summary:
                    issue = iss
                    break
            else:
                issue = self._issues[0]
            new_components = set(c.name for c in issue.components) | set(components)

            if len(new_components) != len(issue.components):
                issue.update(components=list(new_components))
            if deadline and issue.deadline != deadline:
                issue.update(deadline=deadline)

            if issue.status.key != 'closed':
                print u"Issue already exists: https://st.yandex-team.ru/%s" % issue.key
                return

            print "Issue %s closed. Reopen and comment" % issue.key
            issue.update(assignee={})
            issue.transitions['reopen'].execute()
            return issue.comments.create(text=description)


class VaultCertificates(object):
    tag_name = 'certificate'

    def __init__(self, yav_token):
        self.yav = Production(authorization='OAuth %s' % yav_token)
        self.now = datetime.now()

    def list(self):
        for secret in self.yav.list_secrets(tags=['certificate']):
            secret_uuid = secret['uuid']
            secret_name = secret['name']
            value = self.yav.get_version(secret_uuid, packed_value=True)['value']
            key = 'pem' if 'pem' in value else \
                'key' if 'key' in value else \
                'bundle' if 'bundle' in value else \
                ''
            if not key:
                print('No key for secret %s' % secret_uuid)
                continue
            yield secret_uuid, secret_name, value[key]

    @staticmethod
    def parse_not_after(cert):
        x509 = OpenSSL.crypto.load_certificate(OpenSSL.crypto.FILETYPE_PEM, cert)
        not_after = datetime.strptime(x509.get_notAfter(), '%Y%m%d%H%M%SZ')
        return not_after

    def is_expiring(self, not_after, days):
        return (not_after - self.now).days <= days

    def get_expiring(self, days):
        for secret_uuid, secret_name, cert in sorted(self.list(), key=lambda x: x[1]):
            not_after = self.parse_not_after(cert)
            print secret_name, not_after
            if self.is_expiring(not_after, days):
                yield secret_uuid, secret_name, not_after


if __name__ == '__main__':
    try:
        config = yaml.safe_load(open('/etc/yandex/certificates-check.yaml'))
    except Exception as e:
        print 'Unable to load yaml settings, check that config in YAML format: %s' % e.message
        sys.exit(1)

    days_remains = int(config['days_remains'])
    yav = VaultCertificates(config['yav']['token'])
    followers = get_group_members(config['staff']['token'], config['staff']['department'])

    for secret_uuid, name, not_after in yav.get_expiring(days_remains):
        ticket = ErrorTicket(u'Истекает сертификат %s' % name, followers=followers, token=config['startrek']['token'])
        ticket.create(
            description=u'Сертификат валиден до %s\nhttps://yav.yandex-team.ru/secret/%s' % (
                not_after, secret_uuid
            ),
            components=["Администрирование", "Сертификат"],
            deadline=(not_after.date() - timedelta(days=1)).isoformat(),
        )
