#!/bin/bash

set -u
set -e

function usage () {
cat << EOF
    Usage:
        $0 <http[s]://[user:pass@]host[:port]/uri?args> [HTTP_CODE_OK|default(200) [HTTP_BODY_OK|regex-PCRE] [HTTP_TIMEOUT] ]

    Example:
        $0 "http://localhost/ping"
        $0 "http://localhost/ping" 204
        $0 "http://localhost/ping" 200 "ok /ping"
        $0 "http://localhost/ping" 200 "^ok /[Pp]ing" 15
        $0 "http://root:root@localhost/ping" 200 "ok /ping"
EOF
}

function my_log () {
    date "+[%F %T] $@" > /dev/stderr
}

[[ -z $1 || $# > 4 ]]                                   && { usage; exit 0; }
[[ "$1" == "-h" || "$1" == "--help" || "$1" == "-"* ]]  && { usage; exit 0; }

URL="$1"
HTTP_CODE_OK="${2:-200}"
HTTP_BODY_OK="$3"
HTTP_TIMEOUT="${4:-3}"

CURL=""

function get_curl () {
    local curl_string
    if $(which curl > /dev/null); then
        curl_string="$(which curl) -sD- --connect-timeout $HTTP_TIMEOUT"
    else
        curl_string="$(which wget) -qO- --save-headers --timeout $HTTP_TIMEOUT" || \
          { my_log "ERR: could not find not CURL, neither WGET. Exit."; exit 1; }
    fi
    echo "$curl_string"
}

function get_responce () {
    CURL="$(get_curl)" || { return $?; }
    $CURL "$URL" || \
        { E_CODE=$?; my_log "ERR: curl failed with E_CODE=$E_CODE. CMD: $CURL "$URL""; return $E_CODE; }
}

function parse_responce () {
    local E_CODE=100
    local last_paragraph_line_num

    last_paragraph_line_num=$(echo "$@" | awk '/^HTTP/{c=NR}END{print c}')
    [[ -n $last_paragraph_line_num ]] || \
        { my_log "ERR: not found HTTP-header. E_CODE=$E_CODE. Responce: $@"; return $E_CODE; }

    echo "$@" | sed -rn "${last_paragraph_line_num},\$p"
}

function check_http_code () {
    local E_CODE=101
    local res_code
    res_code=$(echo "$@" | awk '/^HTTP/{print $2}')
    #echo $res_code

    [[ "$res_code" == "$HTTP_CODE_OK" ]] || \
        { my_log "ERR: not ok CODE: '$res_code' != '$HTTP_CODE_OK'. E_CODE=$E_CODE"; exit $E_CODE; }
}

function check_http_body () {
    local E_CODE=102
    local res_body

    # skip this check if check body not given
    [[ -z "$HTTP_BODY_OK" ]] && return 0

    # get all text after first empty line (drop headers now)
    res_body="$(echo "$@" | sed -n '/^[[:space:]]*$/,$p' | grep -v '^[[:space:]]*$')"

    echo "$res_body" | grep -qP "$HTTP_BODY_OK" || \
        { my_log "ERR: not ok BODY: '$res_body' != '$HTTP_BODY_OK'. E_CODE=$E_CODE"; exit $E_CODE; }
}

function main () {
    local res
    local parsed_res

    res="$(get_responce)"                   || exit $?  # these are sub-shells, exit them if any
    parsed_res="$(parse_responce "$res")"   || exit $?  # these are sub-shells, exit them if any
    check_http_code "$parsed_res"
    check_http_body "$parsed_res"
}

main
