#!/bin/bash

function my_log () {
    date "+[%F %T] $@" > /dev/stderr
}

####################
### CONFIG START
###

CWD="$(dirname $(readlink -f $0))"
check_http="bash "${CWD}/http-checker.sh""

URL="http://localhost/ping"
HTTP_CODE_OK=200
HTTP_BODY_OK="ok /ping"
HTTP_TIMEOUT=3

RUNNING=600 # restart not often than once per (10 min)
RETRY_CNT=1 # restart not more, than (3) times at once

#SV_PROCESS_GROUP=$($SUPERVISORCTL status | grep -E "pid $(pgrep java | xargs echo | tr -s ' ' '|')," | cut -d' ' -f1)
SV_PROCESS_GROUP="backend"
SV_START_TIMEOUT=30
SUPERVISORCTL="$(which supervisorctl)"

MAINTENANCE=$(which maintenance)

###
### CONFIG END
####################

function check_maintenance () {
    if [[ -f /maintenance.lock ]]; then
        my_log "INFO: file /maintenance.lock found, exiting"
        exit 0
    fi
}

function service_restart () {
    # skip using 'my_log' here, it's output already wrapped by this function
    echo "maintenance close, then sleep 5 sec"
    ${MAINTENANCE} close
    sleep 5

    echo "supervisorctl restart ${SV_PROCESS_GROUP}, then sleep ${SV_START_TIMEOUT} sec"
    ${SUPERVISORCTL} restart ${SV_PROCESS_GROUP}
    sleep ${SV_START_TIMEOUT}

    ### TODO: make check_alive before open (?)
    echo "maintenance open"
    ${MAINTENANCE} open
}

function check_alive () {
    local res_http_checker

    $check_http "$URL" $HTTP_CODE_OK "$HTTP_BODY_OK" $HTTP_TIMEOUT
    res_http_checker=$?

    case $res_http_checker in
        0       ) return 0 ;;
        10[0-2] ) return $res_http_checker ;;   # custom http-checker E_CODE: 100-102
        *       ) exit   $res_http_checker ;;   # any system fails
    esac
}

function main () {
    local E_CODE=110
    local sv_process_pid
    local sv_process_uptime

    sv_process_pid=$(${SUPERVISORCTL} pid ${SV_PROCESS_GROUP})
    sv_process_uptime=$(ps -o etimes= -p ${sv_process_pid} | tr -d ' ')

    check_maintenance

    if ! check_alive; then
        my_log "INFO: check   command:   $check_http '$URL' $HTTP_CODE_OK '$HTTP_BODY_OK' $HTTP_TIMEOUT"
        my_log "INFO: restart command:   ${SUPERVISORCTL} restart ${SV_PROCESS_GROUP}"

        if [ $sv_process_uptime -le $RUNNING ]; then
            my_log "ERR: app was just started: pid=$sv_process_pid started $sv_process_uptime seconds ago. Do nothing, wait until $RUNNING seconds"
            exit $E_CODE
        else
            my_log "INFO: check failed, trying to restart app"
            for (( rtry_c=0; rtry_c < ${RETRY_CNT}; rtry_c++ )); do
                service_restart 2>&1 | while read -r line; do my_log "INFO: $line"; done
                check_alive && break
                
                my_log "INFO: sleep more 10 sec before next restart try"
                sleep 10
                check_alive && break
            done
            if [[ $? != 0 ]]; then
                my_log "ERR: restart failed, given up after $RETRY_CNT retries"
                exit $E_CODE
            fi
        fi
    fi

}

main
