#!/usr/bin/env python

import re
import sys
from collections import Counter, defaultdict

patterns = {
    'received': re.compile(r'^received message via gRPC '
                           r'method=(?P<method>[^ ]*) '
                           r'status=(?P<status>[^ ]*) '
                           r'request_time_ms=(?P<time>\d+) '
                           r'authority=(?P<authority>[^ ]*)'),
    'send': re.compile(r'^send message via gRPC'),
    'failed': re.compile(r'^failed gRPC request '
                         r'method=(?P<method>[^ ]*) '
                         r'status=(?P<status>[^ ]*) '
                         r'request_time_ms=(?P<time>\d+) '
                         r'authority=(?P<authority>[^ ]*)'),
}

timings = defaultdict(Counter)
statuses = Counter()
authorities = Counter()

for line in sys.stdin:
    line = line.strip()
    if not line:
        continue
    for action, p in patterns.items():
        match = p.search(line)
        if match:
            if action == 'send':
                continue
            group = match.groupdict()
            method = group.get('method', '')
            status = group.get('status', '')
            auth = group.get('authority', '').replace(':', '/')
            time = round(float(group.get('time', 0)) / 1000., 3)

            if auth:
                authorities['authority_{}'.format(auth)] += 1
            timings[method][time] += 1
            statuses['message_{}_{}_{}'.format(action, method, status)] += 1

for action, t_cnt in timings.items():
    times = ['%s@%s' % i for i in t_cnt.items()]
    print('@message_{}_timings {}'.format(action, ' '.join(times)))

for status, count in statuses.items():
    print('{} {}'.format(status, count))

for auth, count in authorities.items():
    print('{} {}'.format(auth, count))
