#!/usr/bin/env python3

import os
from kazoo.client import KazooClient
import requests


class App:
    def __init__(self, zk_hosts, zk_prefix, bucket_size=20):
        self.zk_hosts = zk_hosts
        self.zk_prefix = zk_prefix
        self.bucket_size = bucket_size
        self.zk = KazooClient(hosts=self.zk_hosts)
        self.zk.start()

    @property
    def backends_in_zk(self):
        backends = set()
        for x in range(self.bucket_size):
            path = "%s/%s" % (self.zk_prefix, x)
            r, stat = self.zk.get(path)
            backends.add(r.decode('utf-8'))
        return backends

    def check_alive(self, backend):
        url = 'http://%s/ping' % backend
        try:
            r = requests.get(url, timeout=4.0)
            r.raise_for_status()
        except:
            return False
        return True

    def check(self):
        dead = set()
        for backend in self.backends_in_zk:
            if backend == 'mediator':
                continue
            if not self.check_alive(backend):
                dead.add(backend)
        if dead:
            print("2; dead backends in zk: %s" % ",".join(dead))
        else:
            print("0;OK")


if __name__ == '__main__':
    zk_prefix = os.environ.get('ZK_PREFIX')
    if not zk_prefix:
        zk_prefix = '/telemost/backend'
    app = App(
        os.environ['ZK_HOSTS'],
        zk_prefix=zk_prefix
    )
    app.check()
