#!/usr/bin/env python
# -*- coding: utf-8 -*-

import sys
import os
import time
import json
import requests
import hashlib
import logging
import subprocess

logger = logging.getLogger(__name__)


class ServiceDiscoveryError(Exception):
    pass


class ServiceDiscovery:
    BASE_URL = 'http://sd.yandex.net:8080'
    CLIENT_NAME = 'telemost-sd'

    def __init__(self, endpoint, dc, only_ready=False):
        self.endpoint = endpoint
        self.dc = dc
        self.only_ready = only_ready

    def get_data(self):
        data = {
            'cluster_name': self.dc,
            'endpoint_set_id': self.endpoint,
            'client_name': self.CLIENT_NAME,
        }
        return data

    def request(self):
        url = self.BASE_URL + '/resolve_endpoints/json'
        data = self.get_data()
        resp = requests.get(url, data=json.dumps(data), timeout=2)
        resp.raise_for_status()
        return resp.json()

    def resolve(self):
        sd = self.request()
        services = []
        endpoint_set = sd.get('endpoint_set')
        if not endpoint_set:
            raise ServiceDiscoveryError(sd)
        if endpoint_set['endpoint_set_id'] != self.endpoint:
            raise ServiceDiscoveryError(sd)
        for endpoint in endpoint_set['endpoints']:
            if self.only_ready and not endpoint['ready']:
                continue
            services.append(endpoint)
        return services


class App:
    CACHE_FILE = '/tmp/.discovery.cache'

    def __init__(self, dcs, enpoints, xmpp_domain, muc_nickname,
                 jvb_auth_password, jvb_config_path, jvb_hostname):
        self.dcs = dcs
        self.enpoints = enpoints
        self.services, self.services_hash = self.get_services()
        self.xmpp_domain = xmpp_domain
        self.muc_nickname = muc_nickname
        self.jvb_auth_password = jvb_auth_password
        self.jvb_config_path = jvb_config_path
        self.jvb_hostname = jvb_hostname

    def _get_hash(self, s):
        return hashlib.sha256(s).hexdigest()

    def get_services(self):
        services = []
        for dc in self.dcs:
            for endpoint in self.enpoints:
                sd = ServiceDiscovery(endpoint, dc)
                try:
                    services.extend(sd.resolve())
                except ServiceDiscoveryError:
                    pass
        logger.debug('got services: %s' % services)
        services = [_['fqdn'] for _ in services]
        services = sorted(set(services))
        return services, self._get_hash("".join(services))

    @property
    def need_update(self):
        try:
            with open(self.CACHE_FILE, 'r') as fp:
                saved_hash = fp.read()
        except IOError:
            return True
        return saved_hash != self.services_hash

    def write(self):
        logger.info('update config')
        with open(self.jvb_config_path, 'w') as fp:
            fp.write(self.generate_config())

    def get_colibri_stats(self):
        try:
            url = 'http://localhost:8080/colibri/stats'
            resp = requests.get(url, timeout=2)
            resp.raise_for_status()
        except (
            requests.exceptions.ConnectionError,
            requests.exceptions.ConnectTimeout,
            requests.exceptions.HTTPError,
        ):
            return None
        return resp.json()

    @property
    def is_connected(self):
        data = self.get_colibri_stats()
        if not data:
            return False
        if data['mucs_configured'] != len(self.services):
            return False
        # нельзя реагировать рестартом на любую потерю одного конекта
        if data['mucs_joined'] < (data['mucs_configured'] / 2.0 - 1):
            return False
        return True

    @property
    def has_active_conference(self):
        data = self.get_colibri_stats()
        if not data:
            return False
        if data['conferences'] > 0:
            return True
        # if data['participants'] > 10:
        #     return True
        return False

    @property
    def last_update(self):
        return os.stat(self.CACHE_FILE).st_mtime

    @property
    def is_flap(self):
        data = self.get_colibri_stats()
        if not data:
            return False
        if not self.need_update:
            return False
        # если сконфигурировано больше, то это скорее всего деплой xmpp
        # TODO: ловить случаи уменьшения кол-ва xmpp
        if data['mucs_configured'] > len(self.services):
            return True
        return False

    def restart(self):
        cmd = [
            "supervisorctl", "signal", "HUP", "backend"
        ]
        r = subprocess.call(cmd)
        logger.info('restart retval: %d', r)

    def need_run(self):
        if not self.services:
            logger.error('no services from endpoint')
            return False
        if self.has_active_conference:
            logger.warning('has active conferences')
            return False
        if self.is_flap:
            logger.warning('flap detected')
            return False
        if not self.is_connected:
            logger.info('not connected to mucs')
            return True
        if self.need_update:
            logger.warning('need update config')
            return True
        return False

    def run(self):
        if self.need_run():
            self.write()
            self.restart()
            with open(self.CACHE_FILE, 'w') as fp:
                fp.write(self.services_hash)
        else:
            logger.info('up to date')

    def generate_config(self):
        ctx = {
            'XMPP_DOMAIN': self.xmpp_domain,
            'JVB_AUTH_PASSWORD': self.jvb_auth_password,
            'MUC_NICKNAME': self.muc_nickname,
            'JVB_HOSTNAME': self.jvb_hostname,
        }

        config = """org.ice4j.ice.harvest.DISABLE_AWS_HARVESTER=true
org.jitsi.videobridge.rest.jetty.port=9090
org.jitsi.videobridge.rest.COLIBRI_WS_TLS=true
org.jitsi.videobridge.rest.COLIBRI_WS_DOMAIN=%(JVB_HOSTNAME)s:443
org.jitsi.videobridge.ENABLE_REST_SHUTDOWN=true
""" % ctx
        template = """
org.jitsi.videobridge.xmpp.user.%(shard)s.HOSTNAME=%(XMPP_SERVER)s
org.jitsi.videobridge.xmpp.user.%(shard)s.DOMAIN=auth.%(XMPP_DOMAIN)s
org.jitsi.videobridge.xmpp.user.%(shard)s.USERNAME=jvb
org.jitsi.videobridge.xmpp.user.%(shard)s.PASSWORD=%(JVB_AUTH_PASSWORD)s
org.jitsi.videobridge.xmpp.user.%(shard)s.MUC_JIDS=JvbBrewery@internal.auth.%(XMPP_DOMAIN)s
org.jitsi.videobridge.xmpp.user.%(shard)s.MUC_NICKNAME=%(MUC_NICKNAME)s
org.jitsi.videobridge.xmpp.user.%(shard)s.DISABLE_CERTIFICATE_VERIFICATION=true
    """
        for service in self.services:
            ctx['shard'] = service.split('.')[0]
            ctx['XMPP_SERVER'] = service
            config = config + template % ctx
        return config


if __name__ == '__main__':
    logging.basicConfig(
        stream=sys.stdout, level=logging.INFO,
        format='%(asctime)s [%(levelname)s] [%(module)s:%(funcName)s] %(message)s'
    )
    logging.getLogger(__name__).setLevel(logging.DEBUG)
    DCS = [s.strip() for s in
           os.getenv('XMPP_DCS', 'sas,vla,man').split(',')]
    ENDPOINSTS = [s.strip() for s in
                  os.getenv('XMPP_ENDPOINTS').split(',')]

    ipv4_address = os.environ['IPV4_ADDRESS']
    jvb_ws_domain = os.environ['JVB_WS_DOMAIN']
    jvb_hostname = '%s.%s' % (ipv4_address.replace('.', '-'), jvb_ws_domain)

    app = App(
        DCS, ENDPOINSTS, os.environ['XMPP_DOMAIN'].strip(),
        os.environ['MUC_NICKNAME'].strip(),
        os.environ['JVB_AUTH_PASSWORD'].strip(),
        os.environ['JVB_CONFIG'].strip(),
        jvb_hostname,
    )
    app.run()
