#!/usr/bin/python
import argparse
from collections import defaultdict

WEIGHTS_FILE = '/controls/traffic_control.weights'
SERVICE_LIST = ['bygeo']
DEFAULT_WEIGHTS = {'sas' : 1, 'man' : 0, 'vla' : 1, 'myt' : 1, 'iva' : 1}

def write_config(raw_weights):
    with open(WEIGHTS_FILE, 'w') as f:
        f.write(raw_weights)

def format_config(data):
    raw_data = []
    for service, dc_weights in data.items():
        for dc, weight in dc_weights.items():
            raw_data.append('{}_{}, {}'.format(service, dc, weight))
    return '\n'.join(raw_data)

def get_curent_weights():
    current_weights = defaultdict(dict)
    try:
        with open(WEIGHTS_FILE, 'r') as f:
            contents = f.readlines()
    except IOError:
        contents = ''
        print('No config present. Using empty.')

    if contents:
        for line in contents:
            service, dc_weight = line.rstrip().split('_')
            dc, weight = dc_weight.split(',')
            current_weights[service][dc] = weight.strip()
    return current_weights

def set_new_weights(services, dc, action):
    weights = get_curent_weights()
    print('Current weights:')
    print('\n'.join(['{} = {}'.format(k, v) for k, v in weights.items()]))

    for service in services:
        weights[service] = weights.get(service, DEFAULT_WEIGHTS.copy())
        weights[service][dc] = 0 if action == 'close' else 1

    raw_data = format_config(weights)
    write_config(raw_data)
    print('New weights:')
    print('\n'.join(['{} = {}'.format(k, v) for k, v in weights.items()]))


if __name__ == "__main__":
    parser = argparse.ArgumentParser(description='Close and open mediator upstream services by modifying {} file.'.format(WEIGHTS_FILE))

    group = parser.add_mutually_exclusive_group(required=True)
    group.add_argument('--service', choices=SERVICE_LIST, action='append',
                        dest='services',
                        help='To close jicofo service use *group*. Multiple services can be specified by providing --service multiple times.')
    group.add_argument('--all', action='store_const',
                        dest='services',
                        const=SERVICE_LIST,
                        help='Close all possible services in DC.')

    parser.add_argument('--dc', choices=DEFAULT_WEIGHTS.keys(), required=True)
    parser.add_argument('action', choices=['open', 'close'],
                        help='Set weight to 0 in close action, or 1 in open action for specified DC.')

    args = parser.parse_args()

    set_new_weights(args.services, args.dc, args.action)
