#!/usr/bin/python
import os
import shutil
import uuid

import porto
import requests
import uwsgi
from flask import Flask, jsonify, request


DEPLOY_BOX_ID = os.getenv('DEPLOY_BOX_ID', '')
SELF_NAME = 'pod_agent_box_%s/' % DEPLOY_BOX_ID
CONTAINER_PREFIX = "translator"

DEPLOY_POD_PERSISTENT_FQDN = os.getenv('DEPLOY_POD_PERSISTENT_FQDN')
PERSISTENT_FQDN = DEPLOY_POD_PERSISTENT_FQDN.replace('.yp-c.yandex.net', '')
VOLUMES_ROOT = '/srv/'
ENTRY_POINT = '/usr/sbin/disk-init'
DS_LAYER = 'telemost-translator'
DS_ENV = ';'.join(
    '%s=%s' % (key, os.getenv(key))
    for key in [
        'ENVIRONMENT',
        'NGINX_ACCESS_TSKV_LOG_FORMAT',
        'TZ',
        'YANDEX_ENVIRONMENT',
        'DEPLOY_STAGE_ID',
        'DEPLOY_PROJECT_ID',
        'DEPLOY_UNIT_ID',
        'DEPLOY_BOX_ID',
    ]
)
APP_PATH = '/app/telemost-broadcaster.tar.gz'
APP_REMOTE_PATH = '/app/telemost-broadcaster.tar.gz'

NGINX_PORT_START = 800
CONTAINER_PORT_START = 8000
POSSIBLE_CONTAINER_NUMS = set(range(1, 11))
PING_TIMEOUT = 1


class State:
    UNDEFINED = 'undefined'
    STOPPED = 'stopped'
    RUNNING = 'running'
    DEAD = 'dead'
    META = 'meta'
    STARTING = 'starting'
    STOPPING = 'stopping'
    RESPAWNING = 'respawning'
    PAUSED = 'paused'
    DESTROYED = 'destroyed'
    MISSING = 'missing'


app = Flask(__name__)
conn = porto.Connection(timeout=3)
conn.connect()


def generate_uniq():
    return uuid.uuid4().hex[:10]


def actual_name(name):
    return SELF_NAME + name


@app.route('/ping')
def ping():
    return 'pong'


def destroy(input_kwargs):
    name = input_kwargs.get('name')
    if name is None:
        return uwsgi.SPOOL_IGNORE

    container_num = int(name.split('_')[-1])
    translator_port = CONTAINER_PORT_START + container_num
    try:
        requests.put('http://localhost:%s/internal/cluster/inactive' % translator_port, timeout=50)
    except Exception as e:
        app.logger.info('name=%s\tmessage=%s' % (name, 'cant stop sessions: %s' % e))

    volume_path = "/srv/%s" % name.split('/')[-1]
    try:
        volume = conn.FindVolume(volume_path)
    except Exception as e:
        app.logger.info('name=%s\tmessage=%s' % (name, 'volume was not found: %s' % e))

    try:
        conn.Destroy(name)
        app.logger.info('name=%s\tmessage=%s' % (name, 'container destroyed'))
    except Exception as e:
        app.logger.info('name=%s\tmessage=%s' % (name, 'container was not destroyed: %s' % e))
    try:
        volume.Unlink()
        app.logger.info('name=%s\tmessage=%s' % (name, 'volume unlinked'))
    except Exception as e:
        app.logger.info('name=%s\tmessage=%s' % (name, 'volume was not unlinked: %s' % e))
    try:
        os.rmdir(volume_path)
        app.logger.info('name=%s\tmessage=%s' % (name, 'directory removed'))
    except Exception as e:
        app.logger.info('name=%s\tmessage=%s' % (name, 'directory was not removed: %s' % e))

    return uwsgi.SPOOL_OK


uwsgi.spooler = destroy


def get_office_status(port):
    try:
        result = requests.get('http://localhost:%s/ping' % port, timeout=PING_TIMEOUT)
    except Exception:
        return 'FAIL'
    if (result is not None) and (result.status_code == 200):
        return 'OK'
    else:
        return 'FAIL'


@app.route('/porto/<name>', methods=['GET', 'DELETE'])
def container_exists(name):
    container_name = actual_name(name)
    try:
        state = conn.GetProperty(container_name, "state")
    except Exception:
        return jsonify(state=State.MISSING, office='FAIL', port=0)
    else:
        if request.method == 'GET':
            container_num = int(name.split('_')[-1])
            return jsonify(
                state=state,
                office=get_office_status(NGINX_PORT_START + container_num),
                port=NGINX_PORT_START + container_num,
            )
        elif request.method == 'DELETE':
            uwsgi.spool(dict(name=container_name))
            return jsonify(state=State.DESTROYED)


def list_container_names():
    return conn.List(mask=SELF_NAME + CONTAINER_PREFIX + '*')


def list_containers():
    return [name.replace(SELF_NAME, '') for name in list_container_names()]


@app.route('/list')
def get_list():
    return jsonify(list_containers())


@app.route('/list_endpoints')
def get_list_endpoints():
    container_names = list_container_names()
    endpoints = {}
    for container_name in container_names:
        container_num = int(container_name.split('_')[-1])
        port = NGINX_PORT_START + container_num
        endpoints[DEPLOY_POD_PERSISTENT_FQDN + ":%s" % port] = container_name.replace(SELF_NAME, '')
    return jsonify(endpoints)


def generate_env(container_num):
    env = {
        'COMPANY_NAME': 'Yandex',
    }

    default_ports = {
        'DB_PORT': 5432,
        'REDIS_SERVER_PORT': 6379,
        'RABBITMQ_SERVER_URL': 5672,
        'NGINX_PORT': NGINX_PORT_START,
        'DOCSERVICE_PORT': 8000,
        'SPELLCHECKER_PORT': 8080,
        'NGINX_HOOKS_PORT': 3999,
        'NGINX_LOGGIVER_PORT': 3132,
        'FILTER_PROXY_PORT': 8200,
    }
    for env_name, port in default_ports.items():
        new_port = port + container_num
        if env_name == 'RABBITMQ_SERVER_URL':
            new_port = 'amqp://guest:guest@localhost:%s' % new_port
        env[env_name] = new_port

    return ';'.join('%s=%s' % i for i in env.items())


@app.route('/new_container', methods=['POST'])
def new_container():
    uwsgi.lock()
    containers_nums = set(int(n.split('_')[-1]) for n in list_containers())
    container_num = None
    try:
        container_num = (POSSIBLE_CONTAINER_NUMS - containers_nums).pop()
    except KeyError:
        uwsgi.unlock()
        return jsonify(state='full')

    name = CONTAINER_PREFIX + '_' + PERSISTENT_FQDN + '_' + generate_uniq() + '_' + str(container_num)
    volume_path = VOLUMES_ROOT + name
    app.logger.info('name=%s\tmessage=%s' % (name, 'create volume directory'))
    os.makedirs(volume_path)
    app.logger.info('name=%s\tmessage=%s' % (name, 'create volume'))
    conn.CreateVolume(path=volume_path, layers=[DS_LAYER])
    app.logger.info('name=%s\tmessage=%s' % (name, 'create container'))
    container = conn.Create(actual_name(name))
    uwsgi.unlock()
    properties = {
        'root': volume_path,
        'user': 'root',
        'command': ENTRY_POINT,
        'env': DS_ENV + ';' + generate_env(container_num),
    }
    app.logger.info('name=%s\tmessage=%s' % (name, 'set properties'))
    for k, v in properties.items():
        app.logger.info('name=%s\tmessage=%s' % (name, 'set property %s' % k))
        container.SetProperty(k, v)
    app.logger.info('name=%s\tmessage=%s' % (name, 'start container'))
    container.Start()
    shutil.copyfile(APP_PATH, volume_path + APP_REMOTE_PATH)
    return jsonify(name=name, port=NGINX_PORT_START + container_num)


if __name__ == '__main__':
    app.run(debug=True)
