#!/usr/bin/env python
# -*- coding: utf-8 -*-


import sys
import os
import xmpp
import uuid
import time


class FocusCheckException(Exception):
    pass


class FocusCheck:
    def __init__(self, domain, username, password, host='localhost', port=5222,
                 debug=0):
        self.ready = False
        self.last_message_id = 0
        self.username = username
        self.domain = domain
        self.password = password
        self.host = host
        self.port = port
        self.debug = debug
        self.jabber = self.connect()
        self.focus_reply = None

    def connect(self):
        jabberid = "%s@%s" % (self.username, self.domain)
        jid = xmpp.protocol.JID(jabberid)

        connection = xmpp.Client(jid.getDomain(), debug=self.debug)
        connection.connect(server=(self.host, self.port))
        connection.auth(user=jid.getNode(), password=self.password,
                        resource=jid.getResource())
        connection.sendInitPresence(requestRoster=0)
        connection.RegisterHandler('iq', self.make_callback())
        return connection

    def make_callback(self):
        def callback(conn, iq_node):
            node = str(iq_node)
            if 'xmlns="http://jitsi.org/protocol/focus"' in node:
                if 'ready="true"' in node:
                    self.ready = True
                    self.focus_reply = node
            return

        return callback

    def _send_stanza(self, stanza):
        r = self.jabber.send(stanza)
        self.last_message_id = int(r)

    def disco(self):
        stanza = xmpp.Iq(to=self.domain, typ='get')
        query = xmpp.Node('services')
        query.setNamespace('urn:xmpp:extdisco:1')
        stanza.addChild(node=query)
        self._send_stanza(stanza)

    def focus(self):
        stanza = xmpp.Iq(to=self.domain, typ='get')
        query = xmpp.Node('query')
        query.setNamespace(xmpp.protocol.NS_DISCO_ITEMS)
        stanza.addChild(node=query)
        self._send_stanza(stanza)

    def focus2(self):
        """
        <iq id="df018f66-59da-4ed9-958a-36d7de32d0ed:sendIQ" to="focus.telemost.yandex.net" type="set" xmlns="jabber:client">
        <conference
          machine-uid="bb4e67ebda51335ad2a7cf67a8213b41"
          room="3eb1455221c646c1aa592aa25f1f491d@conference.telemost.yandex.net"
          xmlns="http://jitsi.org/protocol/focus">
            <property name="disableRtx" value="false"/>
            <property name="enableLipSync" value="false"/>
            <property name="openSctp" value="true"/>
        </conference></iq>
        :return:
        """
        stanza = xmpp.Iq(to='focus.%s' % self.domain, typ='set')
        query = xmpp.Node('conference')
        query.setNamespace('http://jitsi.org/protocol/focus')
        query.setAttr('machine-uid', 'localchecker')
        query.setAttr('room', 'testroomhealth-%s@conference.%s' % (uuid.uuid4(), self.domain))
        prop1 = xmpp.Node('property')
        prop1.setAttr('disableRtx', 'false')
        query.addChild(node=prop1)
        prop2 = xmpp.Node('property')
        prop2.setAttr('enableLipSync', 'false')
        query.addChild(node=prop2)
        prop3 = xmpp.Node('property')
        prop3.setAttr('openSctp', 'true')
        query.addChild(node=prop3)
        stanza.addChild(node=query)
        self._send_stanza(stanza)

    def check(self):
        self.disco()
        self.focus()
        self.focus2()
        time.sleep(0.5) # wait for jicofo proceed
        for x in range(self.last_message_id // 2):
            self.jabber.Process(1)
        if not self.ready:
            raise FocusCheckException(self.focus_reply)


if __name__ == '__main__':
    check = FocusCheck(
        os.environ['XMPP_DOMAIN'],
        os.environ['AUTH_USER_UNLIM'],
        os.environ['AUTH_PASSWORD_UNLIM'],
        os.environ['XMPP_HOST']
    )
    check.check()
    sys.exit(0)
