# -*- coding: utf-8 -*-

import socket
from os import getenv


def fqdn(errback, getFQDNFromConfig=None, fatal_flag=True):
    banned = ['any.yandex.ru']

    provider_list = []
    if getFQDNFromConfig is not None:
        provider_list.append(getFQDNFromConfig)
    provider_list.extend(PROVIDER_LIST)

    for provider in provider_list:
        errprefix = "Cannot determine my FQDN via %s" % provider.__name__
        try:
            hostname = provider()
        except Exception as e:
            errback("%s, exception:" % (errprefix), e)
            continue

        hostname = hostname.strip(".").lower()

        if len(hostname.split('.')) < 3:
            errback("%s, got \"%s\"" % (errprefix, hostname), None)
            continue

        if hostname in banned:
            errback("%s, got banned hostname \"%s\"" % (errprefix, hostname), None)
            continue

        return hostname

    if fatal_flag:
        raise RuntimeError("Cannot determine my FQDN")
    else:
        return None


########################################################################
# Private

def getQloudInstanceName():
    if getenv('JUGGLER_USE_PERSISTENT_DNS', 'false').lower() == 'true':
        return getenv('QLOUD_DISCOVERY_INSTANCE', '')
    return ''


def getfqdn(*args, **kwargs):
    hostname = socket.getfqdn(*args, **kwargs)
    for af in (socket.AF_INET, socket.AF_INET6):
        try:
            socket.inet_pton(af, hostname)
            raise RuntimeError("Got IP \"%s\" instead of hostname" % hostname)
        except socket.error:
            pass
    return hostname


def fqdnFromDefaultRouteIPv4():
    # 192.0.2.0/24 is reserved for documentation per RFC5737 and/or RFC5735.
    return fqdnFromExternalIp(socket.AF_INET, '192.0.2.1')


def fqdnFromDefaultRouteIPv6():
    # 2001:db8::/32 is reserved for documentation per RFC3849
    return fqdnFromExternalIp(socket.AF_INET6, '2001:db8::1')


def fqdnFromExternalIp(address_family, dest_addr):
    sock = socket.socket(address_family, socket.SOCK_DGRAM)
    try:
        # I use this hack to get default outgoing IP for default route.
        # `9` is port for `discard` service - it does not matter.
        sock.connect((dest_addr, 9))
        outgoing_ip = sock.getsockname()[0]
        return getfqdn(outgoing_ip)
    finally:
        sock.close()


def fqdnFromDns():
    return socket.gethostbyaddr(socket.gethostname())[0]


def fqdnGAI_AF_INET():
    return fqdnGAI(socket.AF_INET)


def fqdnGAI_AF_INET6():
    return fqdnGAI(socket.AF_INET6)


def fqdnGAI_AF_UNSPEC():
    return fqdnGAI(socket.AF_UNSPEC)


def fqdnGAI(address_family):
    # emulate behaviour of `hostname -f` from linux-utils
    return socket.getaddrinfo(socket.gethostname(), None, address_family,
                              socket.SOCK_DGRAM, 0, socket.AI_CANONNAME)[0][3]


PROVIDER_LIST = [
    getQloudInstanceName,
    socket.gethostname,
    fqdnGAI_AF_UNSPEC,
    fqdnGAI_AF_INET,
    fqdnGAI_AF_INET6,
    getfqdn,
    fqdnFromDefaultRouteIPv4,
    fqdnFromDefaultRouteIPv6,
    fqdnFromDns
]
