# -*- coding: utf-8 -*-

"""
    yandex-disk-mongodb-rs-staletag
    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    MongoDB replicaset monitor, which takes info about RS from /etc/mongodb.conf
    Daemon works only with PRIMARY, periodically get rs.status and marks staled replicas
    with tag "lagging" => "true" (string value, important) and otherwise, put replica
    lagging tag back if replication lag has gone.
"""

import os
import sys
import copy
import time
import logging
import logging.config
import argparse
import signal
from multiprocessing.managers import State as ProcessState

import yaml
from pymongo import MongoReplicaSetClient

from .utils.misc import Lockfile, State
from .utils.errors import FlockException
from .utils.logging import CONSOLE_LOGGING

logger = logging.getLogger('mongodb-rs-staletag')
LOCK_FILE = '/var/tmp/mongodb-rs-staletag.lock'


def run():
    parser = argparse.ArgumentParser(
        description='MongoDB ReplicaSet monitor that add "lagging" tag to staled replicas',
        epilog='Daemon should be singleton executable for avoid of races while reconfiguring master'
    )
    arguments = parser.add_argument_group('named arguments')
    arguments.add_argument(
        '--config', '-c',
        type=str, dest='config', required=False, metavar='<config>', default='/etc/mongodb.conf',
    )
    arguments.add_argument(
        '--interval', '-i',
        type=int, dest='interval', required=False, metavar='<interval>', default=1,
        help='Interval between queries to master for checking for lagginess',
    )
    arguments.add_argument(
        '--limit', '-l',
        type=int, dest='limit', required=False, metavar='<limit>', default=5,
        help='Lag limit before replica marked as `lagging`',
    )
    arguments.add_argument(
        '--debug', '-d',
        action='store_true', dest='debug', default=True,
        help='Enable debug messages',
    )

    ###
    options = parser.parse_args()
    state = ProcessState()
    state.value = ProcessState.INITIAL

    def stop(signum, frame):
        if state.value != ProcessState.SHUTDOWN:
            state.value = ProcessState.SHUTDOWN
            logger.info('Stopping application')

    signal.signal(signal.SIGINT, stop)
    signal.signal(signal.SIGTERM, stop)

    if options.debug:
        logger.setLevel(logging.DEBUG)

    if not os.path.exists(options.config) or not os.access(options.config, os.O_RDONLY):
        logger.critical('{config} doesn`t exists or not readable. Exiting.'.format(
            config=options.config
        ))
        sys.exit(1)

    try:
        config = yaml.load(open(options.config))

    except Exception:
        logger.critical('Unable to load MongoDB settings, check that config in YAML format. Exiting')
        sys.exit(1)

    try:
        replicaset = config['replication']['replSetName']
        host, port = 'localhost', config['net']['port']

    except KeyError:
        logger.critical('Unable to get MongoDB replication/port settings from config. Exiting')
        sys.exit(1)

    state.value = ProcessState.STARTED
    while state.value != ProcessState.SHUTDOWN:

        # MongoReplicaSetClient is deprecated, but we use old pymongo :(
        connection = MongoReplicaSetClient(
            '{host}:{port}'.format(host=host, port=port),
            replicaSet=replicaset
        )
        # container for lagging replicas that contains value of `lagging` tag
        lagging = dict()

        # analogue of console `rs.status()` command
        status = connection.admin.command('replSetGetStatus')
        if status['myState'] != State.PRIMARY:
            logger.error('Daemon can be started only on PRIMARY node. Exiting')
            sys.exit(1)

        primary = filter(lambda x: x['state'] == State.PRIMARY, status['members'])
        if primary and len(primary) == 1:
            primary = primary.pop()

        else:
            logger.critical('Unable to find PRIMARY node in rs.status() output. Please repair replicaset manually.')
            sys.exit(1)

        # primary always must be "lagging" = "false"
        lagging[primary['name']] = "false"

        # analogue of console `rs.config()` command
        original_rs_config = connection.local.system.replset.find_one()
        modified_rs_config = copy.deepcopy(original_rs_config)

        # for true measurement of replica lag we must compare primary last oplog time mark w/ replicas oplog last time
        primary_optime_ts = time.mktime(primary['optimeDate'].utctimetuple())

        for replica in filter(lambda x: x['state'] == State.SECONDARY, status['members']):
            replica_optime_ts = time.mktime(replica['optime'].as_datetime().utctimetuple())

            if primary_optime_ts - replica_optime_ts >= options.limit:
                logger.warning('Found staled replica => {} // tagging as `lagging`'.format(replica['name']))
                lagging[replica['name']] = "true"

            else:
                # replica ok // lagging => "false"
                lagging[replica['name']] = "false"

        # update rs.config if necessary
        for idx, node in enumerate(modified_rs_config['members']):
            if node['host'] in lagging:
                modified_rs_config['members'][idx]['tags']['lagging'] = lagging[node['host']]

        # reconfigure RS if original & modified configs are different
        if original_rs_config['members'] != modified_rs_config['members']:
            logger.warning('RS config was changed. So reconfiguring cluster.')
            modified_rs_config['version'] += 1
            connection.admin.command('replSetReconfig', modified_rs_config)

        else:
            logger.debug('RS config wasn`t changed, so staled replicas already tagged as `lagging`')

        # release connection
        connection.close()
        time.sleep(options.interval)


def main():
    try:
        logging.config.dictConfig(CONSOLE_LOGGING)
        logger.propagate = False

        with Lockfile(LOCK_FILE):
            run()

    except FlockException:
        logger.critical('Another instance of application is running! Exiting')
        exit(0)


if __name__ == '__main__':
    main()
