package main

import (
	"bufio"
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"log"
	"os"
	"os/exec"
	"strings"
	"time"
)

var logger = log.New(os.Stderr, "", log.Ldate+log.Lmicroseconds+log.Lshortfile)

func getEnv(key, fallback string) string {
	if value, ok := os.LookupEnv(key); ok {
		return value
	}
	return fallback
}

func readCommands() []Command {
	// Predefined commands
	commands := []Command{
		Command{metric: "unistat", exec: "curl -s http://localhost:3199/unistat_stub_status"},
	}

	// debug purpose
	commandsFile := getEnv("LOGGIVER_COMMANDS_PATH", "/etc/yandex/loggiver/loggiver.commands")
	logger.Println("Use commands file:", commandsFile)
	inFile, _ := os.Open(commandsFile)
	defer func() {
		_ = inFile.Close()
	}()
	scanner := bufio.NewScanner(inFile)
	scanner.Split(bufio.ScanLines)
	for scanner.Scan() {
		line := scanner.Text()
		parts := strings.SplitN(line, "\t", 2)
		if len(parts) < 2 {
			logger.Println("Ignore line:", line)
			continue
		}
		commands = append(commands, Command{metric: parts[0], exec: parts[1]})
	}
	return commands
}

func executeCommand(command Command, unistatChannel chan []Metric) {
	// loggiver request with 7sec timeout
	ctx, cancel := context.WithTimeout(context.Background(), 6.5*1000*time.Millisecond)
	defer cancel()
	logger.Println("Execute:", command.exec)
	cmdResponse, err := exec.CommandContext(ctx, "/bin/bash", "-c", command.exec).Output()
	if err != nil {
		logger.Println("Command failed", command.exec, ". Err:", err)
		unistatChannel <- nil
		return
	}
	var metrics []Metric
	if command.metric == "unistat" {
		err := json.Unmarshal(cmdResponse, &metrics)
		if err != nil {
			logger.Println(err, ". cmd return:", string(cmdResponse))
			metrics = nil
		}
	} else {
		for _, responseLine := range bytes.Split(cmdResponse, []byte("\n")) {
			line := strings.TrimSpace(string(responseLine))
			if len(line) < 1 {
				continue
			}
			metric := getMetric(command.metric, line)
			if metric != nil && metric.Name != "" {
				metrics = append(metrics, *metric)
			} else {
				logger.Println("getMetric failed for", command.metric, "with line", line)
			}
		}
	}
	unistatChannel <- metrics
}

func main() {
	var unistatOutput []Metric
	unistatChannel := make(chan []Metric)
	commands := readCommands()
	for _, command := range commands {
		go executeCommand(command, unistatChannel)
	}

	for range commands {
		currentOutput := <-unistatChannel
		if currentOutput != nil {
			unistatOutput = append(unistatOutput, currentOutput...)
		}
	}
	if unistatOutput != nil {
		jsonOutput, err := json.Marshal(unistatOutput)
		if err == nil {
			fmt.Println(string(jsonOutput))
			return
		}
	}
	fmt.Println("[]")
}
