import re


def parse_check_or_none(value):
    if value is None:
        return None
    result = Bound.parse(value)
    if result:
        return result

    result = Interval.parse(value)
    if result:
        return result

    raise ValueError('Illegal check = ' + value)


class Check(object):
    def check(self, value):
        raise NotImplementedError('MUST be implemented in subclassed')

    def check_or_no_data(self, value):
        return self.check(value) if value else 'no data'


class Bound(Check):
    def __init__(self, value, closed, lower=True):
        self.value = value
        self.closed = closed
        self.lower = lower

    @classmethod
    def lower(cls, value, closed=True):
        return cls(value, closed, lower=True)

    @classmethod
    def upper(cls, value, closed=True):
        return cls(value, closed, lower=False)

    @classmethod
    def parse(cls, value):
        m = re.search('^\s*([><])(=?)\s*([0-9.]+)\s*$', value)
        if m is None:
            return None
        base_op = m.group(1)
        eq_sign = m.group(2)
        value = m.group(3)
        return cls(float(value), eq_sign == '=', base_op == '>')

    def check(self, value):
        check_eval = '%.2f %s %.2f' % (value, self.op, self.value)
        valid = eval(check_eval)
        return None if valid else '%.2f %s %.2f' % (value, self.opposite_op, self.value)

    @property
    def base_op(self):
        return '>' if self.lower else '<'

    @property
    def op(self):
        eq_or_empty = '=' if self.closed else ''
        return '%s%s' % (self.base_op, eq_or_empty)

    @property
    def opposite_op(self):
        eq_or_empty = '' if self.closed else '='
        return '%s%s' % ('<' if self.base_op == '>' else '>', eq_or_empty)

    def __repr__(self):
        return '%s %.2f' % (self.op, self.value)


class Interval(object):
    def __init__(self, lower, upper):
        self.lower = lower
        self.upper = upper

    @classmethod
    def parse(cls, value):
        m = re.search('^\s*([([])\s*([0-9.]+)\s*[;,-]\s*([0-9.]+)\s*([)\]])\s*$', value)
        if m is None:
            return None
        left_brace = m.group(1)
        left_bound = m.group(2)
        right_bound = m.group(3)
        right_brace = m.group(4)
        return cls(Bound.lower(float(left_bound), left_brace == '['), Bound.upper(float(right_bound), right_brace == ']'))

    def check(self, value):
        result = self.lower.check(value)
        if result:
            return result

        return self.upper.check(value)

    def __repr__(self):
        left_brace = '[' if self.lower.closed else '('
        right_brace = ']' if self.upper.closed else ')'
        return '%s%.2f, %.2f%s' % (left_brace, self.lower.value, self.upper.value, right_brace)
