# -*- encoding: utf-8 -*-
import datetime
import logging
import requests
import xml.etree.ElementTree as ET

from os import getenv


class YandexCalendarClient(object):
    URL = 'https://calendar.yandex-team.ru/{endpoint}'

    def __init__(self, token, tz='Europe/Moscow'):
        self.token = token
        self.tz_id = tz
        self.logger = logging.getLogger('YandexCalendarClient')
        self.headers = {
            'user-agent': 'disk.admin.monitors.common.YandexCalendarClient (GSID: %s)'
                          % getenv('GSID', '').replace('\n', ' '),
        }

    def get_events(self, limit=1, start_ts=None):
        url = self.URL.format(endpoint='export/rss.xml')
        query_params = {'private_token': self.token, 'tz_id': self.tz_id, 'limit': limit}
        if start_ts:
            query_params['start_ts'] = start_ts
        response = requests.get(url, params=query_params, headers=self.headers)
        response.encoding = 'utf-8'
        calendar = ET.fromstring(response.text.encode('utf-8'))
        xmlns = "http://www.w3.org/2005/Atom"
        for event in calendar.iterfind("{%s}entry" % xmlns):
            title = event.find("{%s}title" % xmlns).text
            start = event.find("{urn:yandex-calendar}start").text
            start_date = datetime.datetime.strptime(start[:19], '%Y-%m-%dT%H:%M:%S').date()
            yield title, start_date

    def get_one_event(self, start_ts=None):
        events = self.get_events(limit=1, start_ts=start_ts)
        for event in events:
            return event
        else:
            if start_ts:
                dt = datetime.datetime.strptime(start_ts, '%Y-%m-%d').date()
            else:
                dt = datetime.datetime.now().date()
            return None, dt

    def get_current_event(self):
        self.logger.info('Get current event')
        return self.get_one_event()

    @staticmethod
    def is_today(date):
        now = datetime.datetime.now()
        return date <= now.date() and now.hour > 10
