# -*- coding: utf-8 -*-
from os import getenv

import logging
import requests

from startrek_client import Startrek

logging.basicConfig()
requests.packages.urllib3.disable_warnings()


class StartrekClient(Startrek):
    disk_user_agent = 'disk.admin.monitors.common.StartrekClient (GSID: %s)' % getenv('GSID', '').replace('\n', ' ')
    url = 'https://st.yandex-team.ru/'

    def __init__(self, **kwargs):
        # Get token here:
        # https://oauth.yandex-team.ru/authorize?response_type=token&client_id=a7597fe0fbbf4cd896e64f795d532ad2
        self.disk_token = getenv('STARTREK_TOKEN')
        super(StartrekClient, self).__init__(useragent=self.disk_user_agent, token=self.disk_token, **kwargs)
        self._connection.session.verify = False

    def get_issues(self, query, **kwargs):
        """Получение тикетов"""
        return list(self.issues.find(filter=query, **kwargs))

    def create_issue(self, **kwargs):
        return self.issues.create(**kwargs)


class TicketError(Exception):
    pass


class NotUniqError(TicketError):
    pass


class AlreadyExistsError(TicketError):
    pass


class NotExistsError(TicketError):
    pass


class RelatedNotExistsError(NotExistsError):
    pass


class StartrekTicket(object):
    """
    Обычный тикет об ошибке
    """
    follower_logins = set()
    is_uniq = True
    ticket_type = {'key': 'task'}

    def __init__(self, summary, assignee=None, followers=[], queue='CHEMODAN', unique=None):
        self.st_client = StartrekClient()
        self.summary = summary
        self.unique_value = unique or u'%s: %s' % (queue, summary)
        self.search_query = {
            'queue': queue,
            'unique': self.unique_value,
        }
        self.assignee_login = assignee
        self._issues = None
        self._issues_in_release = None
        self.issue_dict = {
            'queue': queue,
            'type': self.ticket_type,
            'unique': self.unique_value,
            'followers': [{'login': l} for l in (self.follower_logins | set(followers))],
        }
        self.key = None

    @property
    def issues(self):
        """
        Получить тикеты, попадающие под `search_query`

        Проверяет тикет на соответсвие флагу `is_uniq`
        """
        # возвращает список
        if not self._issues:
            self._issues = self.st_client.get_issues(self.search_query)
            num = len(self._issues)
            if num == 0:
                raise NotExistsError()
        return self._issues

    def is_can_be_created(self):
        # если тикет не должен быть уникальным, то ок
        if not self.is_uniq:
            return True

        try:
            self.issues
        except NotExistsError:
            # не нашли тикет - ok
            return True
        else:
            if len(self.issues) and self.issues[0].summary != self.summary:
                # поиск работает нечетко - тут тикеты левые
                return True
            return False

    def create(self, description='', components=[], deadline=None, tags=[], do_not_comment=False, fix_version=None, **kwargs):
        """Создание тикета"""
        if not isinstance(description, unicode):
            raise TypeError('Bad description')

        attachments = kwargs.pop('attachments', None)

        if self.is_can_be_created():
            issue_dict = self.issue_dict.copy()
            issue_dict['summary'] = self.summary
            issue_dict['description'] = description
            if components:
                issue_dict['components'] = components
            if tags:
                issue_dict['tags'] = tags
            if deadline:
                issue_dict['deadline'] = deadline
            if self.assignee_login:
                issue_dict['assignee'] = {'login': self.assignee_login}
            if fix_version:
                issue_dict['fixVersions'] = fix_version
            if attachments:
                issue_dict['attachments'] = attachments

            issue = self.st_client.create_issue(**issue_dict)
            print "CREATED %s%s" % (self.st_client.url, issue.key)
            self.key = issue.key
            return self.key
        elif not do_not_comment:
            if self.key:
                issue = self.st_client.issues[self.key]
            else:
                issue = self._issues[0]
            new_components = set(c.name for c in issue.components) | set(components)
            if len(new_components) != len(issue.components):
                issue.update(components=list(new_components))

            if deadline and issue.deadline != deadline:
                issue.update(deadline=deadline)

            if fix_version:
                new_versions = set(v.name for v in issue.fixVersions) | {fix_version}
                if len(new_versions) != len(issue.fixVersions):
                    issue.update(fixVersions=list(new_versions))

            if issue.status.key != 'closed':
                print u"Issue already exists: %s%s" % (self.st_client.url, issue.key)
                self.key = issue.key
                return self.key

            print "Issue %s closed. Reopen and comment" % issue.key
            issue.transitions['reopen'].execute()
            issue.comments.create(text=description)
            self.key = issue.key
            return self.key

    def set_description(self, description=u'', key=None):
        """Обновить описание тикета"""
        if not isinstance(description, unicode):
            raise TypeError('Bad description')
        if not self.key and not key:
            raise ValueError('Ticket not created')
        issue = self.st_client.issues[key or self.key]
        issue.update(description=description)
        print 'Set new description for %s%s' % (self.st_client.url, issue.key)


class ErrorTicket(StartrekTicket):
    """
    Обычный тикет об ошибке
    """
    ticket_type = {'key': 'bug'}
