#!/usr/bin/env python2.7
# -*- coding: utf-8 -*-

import datetime
import logging
from os import getenv
from time import time

from common.juggler import send_juggler, ok, crit

from common.pg_client import PgClient

logging.basicConfig(level=logging.INFO)

JUGGLER_HOST = 'disk_bazinga_devops'

SERVICE = 'cronTasks'

KNOWN_UNITS = {'s': 1, 'm': 60, 'h': 3600, 'd': 24 * 3600}


def parse_delta(delta):
    try:
        return float(delta)
    except ValueError:
        return float(delta[:-1]) * KNOWN_UNITS[delta[-1]]


def parse_setup(string):
    return {task: parse_delta(max_delta)
            for task, max_delta in [task_setup.split(':')
                                    for task_setup in string.split(',')]}


def fetch_result(result):
    rows = result.fetchall()
    return {data['taskId']: data['lastSuccessfulJobFinish'] / 1000
            for data in [row[0] for row in rows]}


def aggregate_shards(by_shards_dicts):
    result = {}
    for shard_dict in by_shards_dicts:
        result = dict(result, **shard_dict)
    return result


def fmt_secs(seconds):
    return str(datetime.timedelta(seconds=seconds))


if __name__ == '__main__':
    pg_group = getenv('PG_GROUP', 'disk_diskq_db').strip('%')
    pg_user = getenv('PG_USER')
    pg_pass = getenv('PG_PASS')
    pg_db = getenv('PG_DB', 'diskqdb')
    pg_port = getenv('PG_PORT', '6432')

    setup = parse_setup(getenv('SETUP'))

    service_name = getenv('SERVICE', SERVICE)

    juggler_host = getenv('JUGGLER_HOST', JUGGLER_HOST)

    client = PgClient(pg_group, pg_user, pg_pass, pg_db, pg_port)

    placeholder = '(%s)' % ', '.join(['%s' for _ in setup])

    query = 'select data from cron_task where task in %s;' % placeholder

    args = setup.keys()

    not_executed_recently = {}

    finishs = client.execute((query, args), fetch_result, aggregate_shards)

    now = time()

    for key, max_delta in setup.items():
        if key not in finishs:
            not_executed_recently[key] = 'no previous executions found - not executed yet or history was dropped'
            continue

        delta = now - finishs[key]
        if delta > max_delta:
            not_executed_recently[key] = 'last executed %s ago > %s' % (fmt_secs(delta), fmt_secs(max_delta))

    if not_executed_recently:
        status, text = crit(str(not_executed_recently))
    else:
        status, text = ok()

    send_juggler(juggler_host, service_name, status, text)
