# -*- coding: utf-8 -*-

import os
import re
from urllib import unquote
from cached_property import cached_property
from collections import defaultdict
from datetime import datetime
from os import getenv
import logging

import multiprocessing as mp

from jinja2 import Template

import yt.wrapper as yt
from nile.api.v1 import clusters
from qb2.api.v1 import extractors, filters

__all__ = [
    'DTraceException',
    'UIDTrace',
    'YCRIDTrace',
    'IDDTrace',

]

logger = logging.getLogger(__name__)

YCRID_EXCLUDE = [None, '', 'null', '-']


class DTraceException(Exception):
    pass


def get_token():
    token = None
    if getenv('YQL_TOKEN'):
        return getenv('YQL_TOKEN')
    elif getenv('YT_TOKEN'):
        return getenv('YT_TOKEN')
    elif getenv('DISK_STAT_YQLTOKEN'):
        return getenv('DISK_STAT_YQLTOKEN')
    return token


class YtClientMixin:
    def get_token(self):
        if hasattr(self, 'yql_token'):
            return self.yql_token
        else:
            return get_token()

    @property
    def yt_client(self):
        return yt.YtClient(proxy="hahn", token=self.get_token())

    def read_table(self, table_name, format=yt.YsonFormat()):
        table = []
        for row in self.yt_client.read_table(table_name, format=format):
            table.append(row)
        return table

    def read_table_iter(self, table_name, format=yt.YsonFormat()):
        for row in self.yt_client.read_table(table_name, format=format):
            yield row


class TraceChunk:
    ISOTIME_FORMAT = '%Y-%m-%d %H:%M:%S'
    ISOTIME_FORMAT_MS = '%Y-%m-%d %H:%M:%S,%f'
    LONG_REQUEST_LIMIT = 0.5  # 2 sec

    def __init__(self, row, long_request_limit=LONG_REQUEST_LIMIT):
        self.row = row
        self.long_request_limit = long_request_limit

    def __repr__(self):
        return "[%(ycrid)s] - %(service)s - %(client_type)s - %(code)s - %(time)s" % {
            'ycrid': self.ycrid,
            'service': self.service,
            'client_type': self.client_type,
            'code': self.status_code,
            'time': self.request_time,
        }

    @cached_property
    def time(self):
        return self.date.strftime('%H:%M:%S,%f')

    @cached_property
    def request_message(self):
        message = self.request or self.message or 'empty'
        return message.decode('utf-8', 'replace')

    def trace_format(self):
        context = {
            'date': self.time,
            'service': self.service,
            'request': self.request_message,
            'code': self.status_code,
            'time': self.request_time,
            'row': self.row
        }
        return "[%(date)s][+%(time)s] - %(service)s - %(code)s - %(request)s" % context

    @cached_property
    def message(self):
        message = self.row.get('message')
        if not message:
            return None
        return self.format_message(message)

    @classmethod
    def format_message(cls, message, max_len=300):
        message = re.sub(r"\s+", ' ', message)
        # mongo
        message = re.sub(r"Binary\('.*', 2\)", 'BIN(...)', message)
        message = re.sub(r"^completed ", '', message)
        # pg
        if message.startswith('dbname'):
            message = re.sub("dbname=", "", message)
            message = re.sub('user=.* " /\* uid', '"/* uid', message)
            if len(message) > max_len:
                placeholder = '<TRUNC>'
                message = message[:max_len / 2 - len(placeholder)] + \
                          placeholder + \
                          message[-max_len / 2:]
        return message[:max_len]

    @staticmethod
    def urldecode(text):
        try:
            return unquote(text).decode('utf8', 'replace')
        except Exception:
            return text

    @classmethod
    def detect_service_from_requests(cls, row):
        message = row.get('message')
        if not message:
            return None
        if 'complete' in message and 'mode=' in message:
            try:
                collection = ".".join(message.split()[1].split('.')[:2])
            except:
                collection = 'unknown'
            return 'mongodb[%s]' % collection
        elif 'dbname=' in message and 'host=' in message:
            match = re.search(r"host=(?P<host>[a-z0-9\.]+)", message,
                              re.IGNORECASE)
            hostname = 'unknown'
            if match:
                hostname = match.group('host')
            return 'postgresql[%s]' % hostname
        elif message.split()[0].lower() in ["get", "post", "put"]:
            match = re.search(r".*http(s?)://(?P<netloc>.*?)/.*", message,
                              re.IGNORECASE)
            if match:
                return 'HTTP[%s]' % match.group('netloc')
        elif message.startswith('amqp://'):
            match = re.search(r"amqp://(?P<netloc>.*?)/.*", message,
                              re.IGNORECASE)
            if match:
                return 'AMQP[%s]' % match.group('netloc')
        return None

    @classmethod
    def detect_service(cls, row):
        service = cls.detect_service_from_requests(row)
        if service:
            return service
        if row.get('appname') == 'platform':
            return 'rest'
        elif row.get('appname') == 'disk':
            return 'mpfs'
        elif row.get('appname') == 'djfs-api':
            return 'djfs-api'
        elif row.get('vhost', '').startswith('dataapi'):
            return 'dataapi'
        elif row.get('vhost', '').startswith('sync-dataapi'):
            return 'sync-dataapi'
        elif 'vhost' in row and row.get('vhost'):
            return row.get('vhost')
        else:
            return None

    @classmethod
    def detect_client_type(cls, ycrid):
        if ycrid.startswith('lnx'):
            return 'desktop[linux]'
        elif ycrid.startswith('win'):
            return 'desktop[win]'
        elif ycrid.startswith('mac'):
            return 'desktop[mac]'
        elif ycrid.startswith('rest-'):
            return 'web'
        elif ycrid.startswith('dataapi'):
            return 'dataapi'
        elif ycrid.startswith('rest_andr'):
            return 'android'
        elif ycrid.startswith('rest_ios'):
            return 'ios'
        elif ycrid.startswith('sw'):
            return 'webdav'
        elif '-' not in ycrid:
            return 'sync-dataapi'
        return None

    @cached_property
    def source(self):
        return self.row.get('tskv_format', '')

    @cached_property
    def status_code(self):
        if self.source == 'ydisk-mpfs-requests-log':
            if self.service.lower().startswith('http'):
                value = self.row.get('message', '').rsplit()[-4]
                return value
        return int(self.row.get('status', 0))

    @cached_property
    def request_time(self):
        return float(self.row.get('request_time', 0))

    @cached_property
    def service(self):
        value = 'unknown'
        try:
            value = self.detect_service(self.row) or 'unknown'
        except:
            pass
        return value

    @cached_property
    def client_type(self):
        value = self.detect_client_type(self.ycrid) or 'unknown'
        return value

    @cached_property
    def ycrid(self):
        return self.row['ycrid']

    @cached_property
    def date(self):
        value = None
        date_format_ms = {
            'ydisk-mpfs-requests-log': 'timestamp',
            'ydisk-mpfs-access-log': 'timestamp',
            'ydisk-mpfs-error-log': 'timestamp_raw',
            'java': 'logtime',
        }
        for source, field_name in date_format_ms.items():
            if self.row.get(field_name) and source in self.source:
                value = datetime.strptime(self.row[field_name],
                                          self.ISOTIME_FORMAT_MS)
                break
        if not value:
            value = datetime.strptime(self.row['iso_eventtime'],
                                      self.ISOTIME_FORMAT)
        return value

    @cached_property
    def request_id(self):
        value = self.row.get('request_id', '')
        return value

    @cached_property
    def is_error(self):
        return str(self.status_code).startswith('5') and self.status_code != 507

    @cached_property
    def is_long(self):
        return self.request_time > self.long_request_limit

    @cached_property
    def request(self):
        return self.row.get('request')


class TraceFlow:
    def __init__(self, ycrid, chunks):
        self.ycrid = ycrid
        self.chunks = chunks

    @property
    def first(self):
        return self.chunks[0]

    @property
    def trace(self):
        return self.chunks[1:]

    def format(self):
        def format_repeat(request_time):
            return 'last message repeated %d times; max - %s' % (
                len(request_time),
                max(request_time),
            )

        lines = [self.first.trace_format()]
        request_time = []
        prev_chunk = None
        for chunk in self.trace:
            new_line = chunk.trace_format()
            if prev_chunk and \
                (prev_chunk.request == chunk.request and
                 prev_chunk.message == chunk.message):
                request_time.append(chunk.request_time)
            else:
                if len(request_time) > 0:
                    lines.append(format_repeat(request_time))
                    request_time = []
                lines.append(new_line)
            prev_chunk = chunk
        if len(request_time) > 0:
            lines.append(format_repeat(request_time))
        return lines


class Report(YtClientMixin):
    def __init__(self, access_table, requests_table, error_table, *args,
                 **kwargs):
        self.access_table = self.read_table(access_table)
        self.requests_table = self.read_table_iter(requests_table)
        self.error_table = self.read_table(error_table)
        self.ycrids, self.chunks = self.proceed_access()
        if requests_table:
            self.proceed_requests()

    def proceed_access(self):
        ycrids = []
        chunks_by_ycrid = defaultdict(list)
        # ordered by iso_eventtime
        for row in self.access_table:
            chunk = TraceChunk(row)
            if chunk.ycrid not in ycrids:
                ycrids.append(chunk.ycrid)
            chunks_by_ycrid[chunk.ycrid].append(chunk)
        return ycrids, chunks_by_ycrid

    def proceed_requests(self):
        for row in self.requests_table:
            chunk = TraceChunk(row)
            if chunk.ycrid not in self.ycrids:
                # TODO: something wrong
                continue
            self.chunks.get(chunk.ycrid).append(chunk)

        for ycrid, chunks in self.chunks.iteritems():
            self.chunks[ycrid] = sorted(chunks, key=lambda x: x.date)

    def get_ycrids_by_chunk_attr(self, attr):
        ycrids = set()
        for ycrid, chunks in self.chunks.iteritems():
            for chunk in chunks:
                if getattr(chunk, attr):
                    ycrids.add(ycrid)
                    break
        return ycrids

    @cached_property
    def errors_ycrids(self):
        ycrids = self.get_ycrids_by_chunk_attr('is_error')
        return ycrids

    @cached_property
    def long_ycrids(self):
        ycrids = self.get_ycrids_by_chunk_attr('is_long')
        return ycrids

    def get_context(self, context=None, **kwargs):
        d = {
            'errors_ycrids': self.errors_ycrids,
            'long_ycrids': self.long_ycrids,
            'total_ycrids': self.ycrids,
        }
        if context:
            d.update(context)
        return d

    def traces(self, ycrids=None):
        if ycrids is None:
            ycrids = self.ycrids
        for ycrid in ycrids:
            chunks = self.chunks.get(ycrid)
            if not chunks:
                logger.warning('No data for ycrid %s' % ycrid)
                continue
            trace = TraceFlow(ycrid, chunks)
            yield trace

    def common_as_text(self):
        # TODO: rewrite with jinja2
        lines = [
            'Common stat:',
            '\tTotal requests: %d' % len(self.ycrids),
            '\tRequests with error: %d' % len(self.errors_ycrids),
            '\tLong requests: %d' % len(self.long_ycrids),
        ]
        if self.long_ycrids:
            lines.append('===Long requests detail')
        for ycrid in self.long_ycrids:
            trace = TraceFlow(ycrid, self.chunks.get(ycrid))
            lines.append('====YCRID: %s (%s)' % (ycrid, trace.first.date))
            lines.extend(trace.format())

        if self.errors_ycrids:
            lines.append('===Error requests detail')
        for ycrid in self.errors_ycrids:
            trace = TraceFlow(ycrid, self.chunks.get(ycrid))
            lines.append('====YCRID: %s (%s)' % (ycrid, trace.first.date))
            lines.extend(trace.format())
        return lines

    def as_text(self, ycrids=None):
        # TODO: rewrite with jinja2
        if ycrids is None:
            ycrids = self.ycrids
        lines = []
        for trace in self.traces(ycrids):
            lines.append('====YCRID: %s (%s)' % (trace.ycrid, trace.first.date))
            lines.extend(trace.format())
        return lines

    def get_template(self, name):
        fp = os.path.join(os.path.abspath(os.path.dirname(__file__)), 'templates', name)
        with open(fp, 'r') as f:
            template = Template(f.read())
        return template

    def as_tracker(self, ycrids=None):
        if ycrids is None:
            ycrids = self.ycrids
        lines = []
        template = self.get_template('report-startrek.html')
        for trace in self.traces(ycrids):
            line = template.render(trace=trace)
            lines.append(line)
        return '\n'.join(lines)

    def as_html(self, ycrids=None):
        if ycrids is None:
            ycrids = self.ycrids
        template = self.get_template('report.html')
        html = template.render(traces=list(self.traces(ycrids)))
        return html


class YQLWrapper(object):
    def __init__(self, yql_token=None):
        self.yql_token = yql_token
        if yql_token is None:
            self.yql_token = self.get_token()

    def get_token(self):
        if hasattr(self, 'yql_token'):
            return self.yql_token
        else:
            return get_token()

    @property
    def yt_client(self):
        return yt.YtClient(proxy="hahn", token=self.yql_token)

    @staticmethod
    def get_table_url(name, date):
        return '//home/logfeller/logs/%s/1d/%s' % (name, date)

    def get_yql_client(self):
        return clusters.Hahn(yql_token=self.yql_token).env(
            parallel_operations_limit=16,
            yt_spec_defaults=dict(
                job_io=dict(
                    table_writer=dict(
                        max_row_weight=128 * 1024 * 1024,
                        max_key_weight=256 * 1024,
                    )
                ),
            )
        ).job()

    @staticmethod
    def get_access_fields(table):
        def fetch_uid(uri):
            re_fetch_uid = re.compile('[^a-z]?uid=(?P<uid>[0-9]+)[^0-9]?')
            match = re_fetch_uid.search(uri)
            if match:
                return match.group('uid')
            return '-'

        fields = [
            extractors.log_fields(
                'status', 'tskv_format',
                'iso_eventtime', 'request_time', 'host',
                'vhost'
            ),
            extractors.log_field('ycrid', '').add_hints(type=str),
        ]

        schema_fields = dict(
            status=str,
            tskv_format=str,
            ycrid=str,
            iso_eventtime=str,
            request_time=str,
            host=str,
            vhost=str,
            fetch_uid=str,
        )

        fields_dict = {
            'ydisk-mpfs-access-log': [
                extractors.log_field('uri').rename('request'),
                extractors.custom('fetch_uid', fetch_uid, 'request').add_hints(
                    type=str),
                extractors.log_fields(
                    'unixtime',
                    'appname',
                    'name',
                    'uid',
                    'request_id',
                    'timestamp',
                ),
            ],
            'ydisk-java-access-log': [
                extractors.log_field('rid').rename('request_id'),
                extractors.log_fields(
                    'unixtime',
                    'uid',
                    'request',
                    'logtime',
                ),
                extractors.custom('fetch_uid', fetch_uid, 'request').add_hints(
                    type=str),
            ],
            'disk-front-access-log': [
                extractors.log_field('unixtime_ms').rename('unixtime'),
                extractors.log_fields(
                    'request_id',
                    'upstream_response_time',
                    'upstream_status',
                    'request',
                ),
                extractors.custom('fetch_uid', fetch_uid, 'request').add_hints(
                    type=str),
            ]
        }

        schema_dict = {
            'ydisk-mpfs-access-log': [
                ('uri', str),
                ('appname', str),
                ('name', str),
                ('uid', str),
                ('request_id', str),
                ('unixtime', str),
                ('timestamp', str),
            ],
            'ydisk-java-access-log': [
                ('rid', str),
                ('uid', str),
                ('request', str),
                ('unixtime', str),
            ],
            'disk-front-access-log': [
                ('unixtime', str),
                ('request_id', str),
                ('request', str),
                ('upstream_status', str),
                ('upstream_response_time', str),
            ]
        }

        order_dict = {
            'ydisk-mpfs-access-log': 'timestamp',
            'ydisk-java-access-log': 'logtime',
        }

        fields.extend(fields_dict.get(table, []))
        schema_fields.update(dict(schema_dict.get(table, [])))
        return fields, schema_fields, order_dict.get(table, 'iso_eventtime')

    def get_access_records(self, result_table, date, ycrids_table_path,
                           force=False):
        if not force and self.yt_client.exists(result_table):
            return result_table, None

        tables = [
            'ydisk-mpfs-access-log',
            'ydisk-java-access-log',
            # 'disk-front-access-log',
        ]
        records = []
        job = self.get_yql_client()
        query_filter = [
            filters.not_(
                filters.one_of('ycrid', YCRID_EXCLUDE))
        ]
        ycrids = set(
            [_['ycrid'] for _ in self.yt_client.read_table(ycrids_table_path)])
        query_filter.append(
            filters.one_of('ycrid', ycrids)
        )

        for table in tables:
            table_path = self.get_table_url(table, date)
            fields, schema_fields, order = self.get_access_fields(table)
            src_log = job.table(table_path, weak_schema=schema_fields)
            log_query = src_log.qb2(
                log='generic-yson-log',
                fields=fields,
                filters=query_filter,
            )
            records.append(log_query)

        job.concat(*records).sort('iso_eventtime').put(result_table)
        return result_table, job

    def get_ycrids_by_ycrid(self, result_table, ycrid, force=False):
        if not force and self.yt_client.exists(result_table):
            logger.info('%s exists, return' % result_table)
            return result_table
        self.yt_client.create(
            "table",
            result_table,
            attributes={
                'schema': [{"name": "ycrid", "type": "string"}]
            },
            recursive=True)
        self.yt_client.write_table(result_table, [{"ycrid": str(ycrid)}])
        return result_table

    def get_ycrids_from_idd(self, result_table, yt_path, force=False):
        if not force and self.yt_client.exists(result_table):
            logger.info('%s exists, return' % result_table)
            return result_table
        tables = [
            yt_path + '/all_errors',
            yt_path + '/mpfs_long_requests',
            yt_path + '/java_long_requests',
        ]
        records = []
        job = self.get_yql_client()
        for table in tables:
            src_log = job.table(table, weak_schema={'ycrid': str})
            records.append(
                src_log.qb2(
                    log='generic-yson-log',
                    fields=[
                        extractors.log_field('ycrid', '').add_hints(type=str)
                    ],
                    filters=[
                        filters.not_(
                            filters.one_of('ycrid', YCRID_EXCLUDE))
                    ],
                ).unique('ycrid').project('ycrid')
            )
        job.concat(*records).unique('ycrid').put(result_table,
                                                 schema={'ycrid': str})
        job.run()
        return result_table

    def get_ycrids_by_uid(self, result_table, uid, date, force=False):
        if not force and self.yt_client.exists(result_table):
            logger.info('%s exists, return' % result_table)
            return result_table

        tables = [
            'ydisk-mpfs-access-log',
            'ydisk-java-access-log',
        ]
        records = []
        job = self.get_yql_client()
        for table in tables:
            table_path = self.get_table_url(table, date)
            fields, schema_fields, order = self.get_access_fields(table)
            src_log = job.table(table_path, weak_schema=schema_fields)
            records.append(
                src_log.qb2(
                    log='generic-yson-log',
                    fields=fields,
                    filters=[
                        filters.or_(
                            filters.equals('uid', uid),
                            filters.equals('fetch_uid', uid),
                        ),
                        filters.not_(
                            filters.one_of('ycrid', YCRID_EXCLUDE))
                    ],
                ).unique('ycrid').project('ycrid')
            )
        job.concat(*records).unique('ycrid').put(result_table,
                                                 schema={'ycrid': str})
        job.run()
        return result_table

    def get_requests_records(self, result_table, date, ycrids_table_path,
                             force=False):
        if not force and self.yt_client.exists(result_table):
            logger.info('%s exists, return' % result_table)
            return result_table, None

        def parse_request_time(message):
            time_part = message.rsplit()[-1]
            try:
                return float(time_part)
            except ValueError:
                return 0.0

        job = self.get_yql_client()
        table = 'ydisk-mpfs-requests-log'
        table_path = self.get_table_url(table, date)
        fields = [
            extractors.log_fields(
                'ycrid', 'request_id', 'iso_eventtime', 'message', 'pid',
                'tskv_format', 'timestamp',
            ),
            extractors.custom('request_time', parse_request_time,
                              'message').add_hints(type=float),
        ]
        query_filter = [
            filters.not_(
                filters.one_of('ycrid', YCRID_EXCLUDE))
        ]
        ycrids = set(
            [_['ycrid'] for _ in self.yt_client.read_table(ycrids_table_path)])
        query_filter.append(
            filters.one_of('ycrid', ycrids)
        )
        src_log = job.table(table_path)
        src_log.qb2(
            log='generic-yson-log',
            fields=fields,
            filters=query_filter,
        ).sort('timestamp').put(result_table)
        return result_table, job

    def get_error_records(self, result_table, date, ycrids_table_path,
                          force=False):
        if not force and self.yt_client.exists(result_table):
            logger.info('%s exists, return' % result_table)
            return result_table, None

        def parse_timestamp_raw(message):
            return message.get('timestamp_raw', '')

        job = self.get_yql_client()
        table = 'ydisk-mpfs-error-log'
        table_path = self.get_table_url(table, date)
        fields = [
            extractors.log_fields(
                'ycrid', 'request_id', 'iso_eventtime', 'message', 'pid',
                'tskv_format', 'timestamp', 'appname', '_other'
            ),
            extractors.custom('timestamp_raw', parse_timestamp_raw,
                              '_other').add_hints(type=str),
        ]
        query_filter = []
        ycrids = set(
            [_['ycrid'] for _ in self.yt_client.read_table(ycrids_table_path)])
        query_filter.append(
            filters.one_of('ycrid', ycrids)
        )
        src_log = job.table(table_path)
        src_log.qb2(
            log='generic-yson-log',
            fields=fields,
            filters=query_filter,
        ).sort('timestamp').put(result_table)
        return result_table, job

    def get_uploader_records(self, result_table, date, join_table_path,
                             join_table_field,
                             force=False):
        if not force and self.yt_client.exists(result_table):
            logger.info('%s exists, return' % result_table)
            return result_table, None

        job = self.get_yql_client()
        table = 'ydisk-uploader-log'
        table_path = self.get_table_url(table, date)
        fields = [
            extractors.log_fields(
                'ycrid', 'iso_eventtime', 'message',
                'tskv_format', 'timestamp', 'logtime', 'host', 'class'
            ),
            extractors.log_field('rid').rename('request_id'),
        ]
        join_table = job.table(join_table_path).qb2(
            log='generic-yson-log',
            fields=extractors.log_fields(join_table_field)
        )
        src_log = job.table(table_path)
        src_log.qb2(
            log='generic-yson-log',
            fields=fields,
        ).join(
            join_table, by=(join_table_field,), type='inner',
            assume_small_right=True
        ).sort('timestamp').put(result_table, )
        return result_table, job


def run_yql(method_name, args, yql_token=None):
    yql_wrapper = YQLWrapper(yql_token=yql_token)
    logger.debug('call %s: %s' % (method_name, ';'.join(args)))
    func = getattr(yql_wrapper, method_name)
    try:
        table, job = func(*args)
    except Exception, e:
        logger.error('fail %s: %s' % (method_name, e))
        return None
    logger.debug('got table: %s' % table)
    if job:
        logger.debug('have job for %s' % method_name)
        try:
            job.run()
        except Exception, e:
            logger.error('fail run %s: %s' % (method_name, e))
            return None
    return table


def get_tables(args, yql_token=None):
    """
    Функция чтобы делать multiprocess без завязки на классы
    :param args: словарь, где ключ - имя метода в YQLWrapper,
    значение - агрументы к методу. Именованные аргементы не поддерживаются.
    :return: словарь, где ключ - имя метода в YQLWrapper,
    значение - все что вернул метод после выполнения
    """
    pool = mp.Pool(processes=4)
    wait = {}
    for method, arg in args.items():
        r = pool.apply_async(run_yql, (method, arg, yql_token))
        wait[method] = r
    result = {}
    for method, ares in wait.items():
        r = ares.get()
        if not r:
            raise DTraceException('async task return None')
        result[method] = r
    return result


class BaseTrace(YtClientMixin):
    result_prefix = '//home/disk-dev/bpsavelev/dtrace'

    @property
    def yt_client(self):
        return yt.YtClient(proxy="hahn", token=self.yql_token)

    def get_ycrids(self):
        raise NotImplementedError()

    def get_result_table(self, name):
        raise NotImplementedError()

    @property
    def ycrids(self):
        table_name = self.get_ycrids()
        table = set()
        for record in self.yt_client.read_table(table_name,
                                                format=yt.YsonFormat()):
            table.add(record['ycrid'])
        return table

    def get_records(self):
        args = {
            'get_requests_records': [
                self.get_result_table('requests'),
                self.date,
                self.get_ycrids(),
            ],
            'get_access_records': [
                self.get_result_table('access'),
                self.date,
                self.get_ycrids()
            ],
            'get_error_records': [
                self.get_result_table('error'),
                self.date,
                self.get_ycrids()
            ]
        }

        tables = get_tables(args, self.get_token())
        logger.debug(tables)
        records = {
            'access_table': tables.get('get_access_records'),
            'requests_table': tables.get('get_requests_records'),
            'error_table': tables.get('get_error_records'),
        }
        return records

    def report(self):
        report = Report(**self.get_records())
        return report


class YCRIDTrace(BaseTrace):
    def __init__(self, ycrid, date, yql_token):
        self.ycrid = ycrid
        self.date = date
        self.yql_token = yql_token
        self.yql_wrapper = YQLWrapper(self.yql_token)

    def get_ycrids(self):
        table_name = self.get_result_table('ycrids')
        return self.yql_wrapper.get_ycrids_by_ycrid(
            table_name, self.ycrid)

    def get_result_table(self, name):
        yt_path = "%s/%s-%s-%s" % (
            self.result_prefix,
            self.date,
            self.ycrid,
            name
        )
        return yt_path


class UIDTrace(BaseTrace):
    def __init__(self, uid, date, yql_token):
        self.uid = uid
        self.date = date
        self.yql_token = yql_token
        self.yql_wrapper = YQLWrapper(self.yql_token)

    def get_result_table(self, name):
        yt_path = "%s/%s-%s-%s" % (
            self.result_prefix,
            self.date,
            self.uid,
            name
        )
        return yt_path

    def get_ycrids(self):
        table_name = self.get_result_table('ycrids')
        return self.yql_wrapper.get_ycrids_by_uid(table_name, self.uid,
                                                  self.date)


class IDDTrace(BaseTrace):
    def __init__(self, idd_yt_path, date, yql_token, yt_root=None):
        self.idd_yt_path = idd_yt_path
        self.yt_root = yt_root
        self.date = date
        self.yql_token = yql_token
        self.yql_wrapper = YQLWrapper(self.yql_token)

    def get_ycrids(self):
        table_name = self.get_result_table('ycrids')
        return self.yql_wrapper.get_ycrids_from_idd(
            table_name, self.idd_yt_path)

    @property
    def result_prefix(self):
        if self.yt_root:
            return self.yt_root
        return self.idd_yt_path

    def get_result_table(self, name):
        yt_path = "%s/%s-dtrace-%s" % (
            self.result_prefix,
            self.date,
            name
        )
        return yt_path
