#!/usr/bin/env python2.7
# -*- coding: utf-8 -*-
import time
from os import getenv

import requests

from nile.api.v1.datetime import Datetime, next_day
from nile.api.v1.statface.client import StatfaceProductionClient
from nile.api.v1.statface.report import StatfaceReport

from disk.admin.monitors.common import html
from disk.admin.monitors.common import mailer


def generate_html_body(dt, sms_table, sms_count, phone_table, phone_count):
    return ("""
        <h2>{phone_count} calls yesterday ({dt})</h2>
        {phone_table}

        <h2>{sms} sms yesterday ({dt})</h2>
        {sms_table}
    """.format(
        dt=dt,
        sms=sms_count,
        sms_table=html.html_table(sms_table, ['host_name', 'service_name']),
        phone_count=phone_count,
        phone_table=html.html_table(phone_table, ['host_name', 'service_name', 'login'])
        )
    )


def publish_stat(username, password, sms, phone, fielddate):
    client = StatfaceProductionClient(username=username, password=password)
    report = StatfaceReport().path('Disk/Admin/Duty-calls').scale('daily')
    data = {
        'sms': sms,
        'fielddate': fielddate,
    }
    for robot_type, calls in phone.items():
        data['phone_' + robot_type] = calls
    report = report.client(client).data([data])
    report.publish()
    print("Report published")


def get_juggler_notifications(yesterday, robot_usernames, sms_username):
    url = "http://juggler-api.search.yandex.net/v2/history/get_notifications"
    yesterday_ts = int(time.mktime(Datetime.from_iso(yesterday).timetuple()))
    phone_escalation_filters = [{"method": "phone_escalation", "login": l} for l in robot_usernames]
    data = {
        "filters": phone_escalation_filters + [{"method": "sms", "login": sms_username}],
        "since": yesterday_ts,
        "until": yesterday_ts + 24 * 3600,
        "page": 1,
        "page_size": 200,
    }
    response = requests.post(url, json=data, headers={
        'user-agent': 'disk.monitors.duty_calls (GSID: %s)' % getenv('GSID', '').replace('\n', ' '),
    }).json()
    if not response:
        return None
    return response['notifications']


def count_sms_notifications(juggler_notifications):
    if not juggler_notifications:
        return 0, []
    events = [e for e in juggler_notifications if e.get("method") == "sms"]
    return len(events), events


def count_phone_notifications(juggler_notifications, robots):
    sums = {robot_type: 0 for robot_type in robots.values()}
    if not juggler_notifications:
        return sums, None
    events = [e for e in juggler_notifications if e.get("method") == "phone_escalation"]
    for event in events:
        robot_type = robots[event['login']]
        sums[robot_type] += 1
    return sums, events


def main():
    stat_username = getenv("STAT_USER")
    stat_password = getenv("STAT_PASSWORD")
    sms_username = getenv("SMS_USERNAME")
    robot_usernames = dict(i.split('=') for i in getenv("ROBOT_USERNAMES").split(','))
    yesterday = next_day(Datetime.now(), offset=-1)
    juggler_notifications = get_juggler_notifications(yesterday, robot_usernames.keys(), sms_username)
    sms_count, sms_table = count_sms_notifications(juggler_notifications)
    phone_count, phone_table = count_phone_notifications(juggler_notifications, robot_usernames)
    print sms_count, phone_count
    mailer.send_email(
        getenv("RECIPIENT"),
        "%s calls, %s sms at %s" % (sum(phone_count.values()), sms_count, yesterday),
        generate_html_body(yesterday, sms_table, sms_count, phone_table, sum(phone_count.values()))
    )
    publish_stat(stat_username, stat_password, sms_count, phone_count, yesterday)
