#!/usr/bin/python
from collections import defaultdict
from email.mime.multipart import MIMEMultipart
from email.mime.text import MIMEText
from os import getenv
from smtplib import SMTP
from urlparse import urljoin

import requests

C_API_URL = 'https://c.yandex-team.ru/api/'
OOPS_URL = 'https://oops.yandex-team.ru/'
OOPS_API_URL = urljoin(OOPS_URL, '/api/')
RACKTABLES_URL = 'https://racktables.yandex.net/'

REQUESTS_HEADERS = {
    'user-agent': 'disk.monitors.host2switch (GSID: %s)' % getenv('GSID', '').replace('\n', ' '),
}


def c_group2hosts(group):
    url = urljoin(urljoin(C_API_URL, 'groups2hosts/'), group)
    response = requests.get(url, headers=REQUESTS_HEADERS)
    hosts = response.content.rstrip().split('\n')
    return hosts


def oops_switch_name(hostname):
    request_path = 'hosts/{}/attributes/vlan_list'.format(hostname)
    url = urljoin(OOPS_API_URL, request_path)
    result = requests.get(url).json()
    try:
        return result['vlans'][0]['switchName']
    except (KeyError, IndexError):
        return '-'


def oops_switch_url(switch):
    return OOPS_URL + '?q=switch%20{}'.format(switch)


def racktables_switch_url(switch):
    return RACKTABLES_URL + '?page=oneshot&realm=object&q=%s' % switch


def oops_host_url(hostname):
    return urljoin(OOPS_URL, 'host/%s' % hostname)


def oops_groups_query(c_groups):
    return '+%7C%7C+'.join(
        'conductor+%s' % c_group
        for c_group in c_groups.split(',')
    )


def send_email(recipient, html_table):
    msg = MIMEMultipart('alternative')
    msg['From'] = 'robot-disk-duty-adm@yandex-team.ru'
    msg['To'] = recipient
    msg['Subject'] = 'Postgres hosts per switch distribution'
    msg.attach(MIMEText(html_table, 'html', "utf-8"))
    server = SMTP('outbound-relay.yandex.net')
    server.sendmail(msg['From'], recipient, msg.as_string())


if __name__ == '__main__':
    recipient = getenv('RECIPIENT')
    c_groups = getenv('C_GROUPS', 'disk_pg,disk_sharpei,disk_pg_srv')

    switches = defaultdict(list)
    for c_group in c_groups.split(','):
        for hostname in c_group2hosts(c_group):
            switches[oops_switch_name(hostname)].append(hostname)

    html_table = [
        "<html>"
        "  <head>",
        "    <style>",
        "      table { border-collapse: collapse; }",
        "      a { color: black; text-decoration: none; }",
        "    </style>",
        "  </head>",
        "  <body>",
        '  <a href="%s/?q=%s+group+switch">details in oops</a>' % (OOPS_URL, oops_groups_query(c_groups)),
        '  <table border="1">',
        '    <tr><th>switch</th><th>host</th></tr>',
    ]
    for i, (switch, hosts) in enumerate(sorted(switches.iteritems(), key=lambda x: len(x[1]), reverse=True)):
        background = '#e6e6e6' if i % 2 else 'white'
        html_table.append(
            '<tr style="background: {background};">'
            '<td rowspan={rowspan}>'
            '<a href="{switch_oops_url}">{switch}</a>'
            ' <a href="{racktables_switch_url}">rt</a>'
            '</td>\n'.format(
                background=background,
                rowspan=len(hosts)+1,
                switch_oops_url=oops_switch_url(switch),
                racktables_switch_url=racktables_switch_url(switch),
                switch=switch,
            )
        )
        for hostname in hosts:
            html_table.append(
                '<tr style="background: {}">'
                '<td><a href="{}">{}</a></td>'
                '</tr>'.format(background, oops_host_url(hostname), hostname)
            )
        html_table.append('</tr>')
    html_table.append('</table></body></html>')
    html_table = "\n".join(html_table)
    send_email(recipient, html_table)
