# -*- coding: utf-8 -*-

import logging
logging.basicConfig(level=logging.DEBUG, format="[%(asctime)s] %(levelname)s %(name)s %(message)s")
logger = logging.getLogger(__name__)

import statface_client
import pandas as pd
import datetime
import os
import re
import time
import sys

# extend path for common lib
from os import path as os_path
dir_path = '/'.join(os_path.abspath(__file__).split('/')[:-2])
sys.path = [dir_path] + sys.path

from common import mailer
from common import html


mail_list = {
    'iOS': ['lizvad', 'asinusov'],
    'android': ['lizvad', 'feelgood']
}
# Run only between specific hours
hours_interval = [9, 21]
# Filter versions with less than X active userss
min_users=1000

platforms = mail_list.keys()
version_threshold = 1.
total_threshold = 1.


class StatCollector(object):

    def __init__(self, path, prod=True, capitalize=False):
        self.stat_token = os.getenv('STAT_TOKEN')
        self.client = statface_client.StatfaceClient(oauth_token=self.stat_token, host=statface_client.STATFACE_PRODUCTION)
        cap = str.capitalize if capitalize else str
        path = '/'.join(map(cap, path.split('/')))
        self.report = self.client.get_report(path)

    def download_data(self, interval, agg='d', **kwargs):
        '''
        :param interval: int or pair of min/max dates
        :return:
        '''
        days, date_min, date_max = None, None, None

        try:
            dt = list(interval)
            date_min, date_max = map(
                lambda x: pd.to_datetime(x).strftime('%Y-%m-%d'),
                dt
            )
        except TypeError:
            days = interval

        if not (date_min and date_max):
            data = self.report.download_data(agg, app='disk', _period_distance=days, **kwargs)
        else:
            data = self.report.download_data(agg, app='disk', date_min=date_min, date_max=date_max, **kwargs)
        return data


def collect_crashes_data(interval=0):
    sc = StatCollector('Disk/Testing/MobileAppCrashes')
    e = None
    for i in range(3):
        try:
            data = pd.DataFrame(sc.download_data(interval))
            return data
        except Exception as e:
            logger.exception(e)
    raise e

# convert text version into numeric for sorting
def numeric_version(ver):
    try:
        sver = ver.split('.')
        major = float(sver[0])
        minor = float('0.' + ''.join([re.sub(r'\D', '', s) for s in sver[1:]]))
        return major + minor
    except:
        return -1

def get_crash_alerts(df, platform):
    _version = 'version_override_by_dictionary'
    _total = '_total_'
    _crash_rate = 'total_crash_rate'
    alerts = []
    vdf = df[(df.platform == platform) & (df.active_users > min_users)][['version', _version, _crash_rate]]
    vdf['n_ver'] = vdf['version'].apply(numeric_version)
    vdf.sort_values(by='n_ver', ascending=False, inplace=True)
    active_versions = list(vdf.nlargest(2, ['n_ver'])['version'].values) + [_total]
    logger.info("{} versions: {}".format(platform, active_versions))

    for index, row in vdf.iterrows():
        ver, t_version, rate = row['version'], row[_version], row[_crash_rate]
        if ver not in active_versions:
            continue
        threshold = total_threshold if ver == _total else version_threshold
        logger.info('Rate is %s for version %s' % (rate, t_version))
        if rate >= threshold:
            alert = {'version': t_version.encode('utf-8'), 'crash rate': '{:.2f}%'.format(rate)}
            alerts.append(alert)
    logger.info("Collected alerts for platform {}: {}".format(platform, alerts))
    return alerts

def send_alerts():
    data = collect_crashes_data(0)
    for platform, mails in mail_list.items():
        alerts = get_crash_alerts(data, platform)
        if not alerts:
            logger.info('No alerts for platform %s' % platform)
            continue

        message = u''
        message += html.html_table(alerts, ['version', 'crash rate'])
        message += u'<br>Alert chart: https://datalens.yandex-team.ru/preview/lr7uikvk0s7eh-mobileappcrashes?platform=%s' % platform

        for mail in mails:
            if not ('@' in mail):
                mail = '%s@yandex-team.ru' % mail
            rep = 0
            max_rep = 2
            while rep < max_rep:
                try:
                    logger.info('Sending mail to %s' % mail)
                    mailer.send_email(mail,
                                      'Disk %s crash rate alert' % platform,
                                      message)
                    time.sleep(5)
                    rep = max_rep
                except Exception as e:
                    rep += 1
                    logger.exception(e)


def is_time_ok():
    dt = datetime.datetime.now()
    run = dt.hour > hours_interval[0] and dt.hour < hours_interval[1]
    if not run:
        logger.info('Wrong time to run')
    return run

if __name__ == '__main__':
    if is_time_ok():
        send_alerts()
