#!/usr/bin/env python3
# -*- encoding: utf-8 -*-

try:
    from infra.yasm.yasmapi import GolovanRequest
except:
    from yasmapi import GolovanRequest
import argparse
import datetime
import time
import math

from disk.admin.monitors.common.juggler import send_juggler

HOST = "ASEARCH"
ITYPE = 'deploy'
STAGE = 'disk-mpfs-stable'
METRIC_NAME = 'unistat-mpfs_nginx_by_handle_handles.billing.service_delete.5xx_ammm'

DEFAULT_PERIOD = 300


def get_signal(metric, stage=STAGE, itype=ITYPE):
    return "itype=%s;stage=%s:or(%s,0)" % (itype, stage, metric)


def get_signal_max_value(signal, time_point, window, period=DEFAULT_PERIOD):
    max_tries = 3
    for n_try in range(max_tries):
        try:
            return sum([value[1].get(signal) for value in
                        GolovanRequest(HOST, period, time_point - window, time_point, [signal])], 0)
        except Exception as e:
            if n_try < max_tries - 1:
                time.sleep(5)
                continue
            raise e


def to_juggler(status, message):
    send_juggler('mpfs.disk.yandex.net', 'handle.billing.service_delete_5xx_day', status, message)


def send_ok():
    to_juggler('OK', 'OK')


def send_crit(message):
    to_juggler('CRIT', message)


def check_alert(threshold, start_hour, full_window):
    signal = get_signal(METRIC_NAME)
    now = datetime.datetime.now()
    start = datetime.datetime.combine(now.today(), datetime.time(start_hour))
    window = full_window - math.ceil(abs(((now - start).total_seconds())))

    now_value = get_signal_max_value(signal, int(start.strftime("%s")), window)

    if now_value is None:
        send_crit('value for golovan signal "%s" not found' % METRIC_NAME)
        return

    if now_value > threshold:
        send_crit('{0} > {1}; {3} - {0} for the last {2} second;'
                  .format(now_value, threshold, window, METRIC_NAME))
        return

    send_ok()


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('-t', '--threshold', default=5, type=int, help='Trigger threshold')
    parser.add_argument('--start_hour', default=11, type=int, help='Hour of the start of inspection')
    parser.add_argument('--end_hour', default=20, type=int, help='Hour of the end of inspection')

    args = parser.parse_args()
    hour = datetime.datetime.now().hour

    if args.start_hour <= hour <= args.end_hour:
        window_in_second = (24 - abs(args.end_hour - args.start_hour)) * 60 * 60
        check_alert(args.threshold, args.start_hour, window_in_second)
    else:
        send_ok()


if __name__ == '__main__':
    main()
