#!/usr/bin/env python
# -*- coding: utf-8 -*-

from collections import Counter
from os import getenv

import logging
import requests

from common import graphite
from common.pg_client import PgClient



def collect_stats(task_stats, shard_num, task_queue, all_metrics):
    aggr_tasks = Counter()
    mpfs_queues = Counter()

    for task in task_stats:
        task_name, task_status, task_count = task
        task_name = task_name.replace('.', '-')
        append_metric("by_shard_%02d_task_%s_%s" % (shard_num, task_name, task_status), task_count, all_metrics)

        aggr_status = 'inwork' if task_status in {'running', 'starting', 'ready'} else task_status

        if task_name.startswith('mpfs-'):
            aggr_name = 'mpfs'
            if task_name in task_queue:
                mpfs_queues['%s_%s' % (task_queue[task_name], aggr_status)] = task_count
        elif any(photoslice_name in task_name for photoslice_name in ['ClusterPlaces', 'Photoslice', 'photoslice']):
            aggr_name = 'photoslice'
        elif 'remindPhotos' in task_name:
            aggr_name = 'lenta-remind'
        elif any(lenta_name in task_name for lenta_name in
                 ['Lenta', 'lenta', 'Memories', 'updateNotificationBlock']):
            aggr_name = 'lenta'
        elif any(cool_lenta_reindex_future_name in task_name for cool_lenta_reindex_future_name in
                     ['processOneIntervalLowPriority']):
            aggr_name = 'cool_lenta_reindex'
        elif any(cool_lenta_name in task_name for cool_lenta_name in
                 ['processOneInterval', 'processOneIntervalHighPriority', 'scheduleProcessOne', 'scheduleMordaRotation']):
            aggr_name = 'cool_lenta'
        elif any(cool_lenta_future_name in task_name for cool_lenta_future_name in
                     ['processOneFutureInterval']):
            aggr_name = 'cool_lenta_future'
        elif any(cool_lenta_rotation_name in task_name for cool_lenta_rotation_name in
                 ['addMordaBlock']):
            aggr_name = 'cool_lenta_rotation'
        elif 'Preview' in task_name:
            aggr_name = 'preview'
        elif 'trashClean' in task_name:
            aggr_name = 'trash_clean'
        elif 'HiddenDataCleaner' in task_name:
            aggr_name = 'hidden_data_clean'
        else:
            aggr_name = 'other'
        aggr_tasks['%s_%s' % (aggr_name, aggr_status)] += task_count
        aggr_tasks['total_%s' % aggr_status] += task_count

    return aggr_tasks, mpfs_queues


def fetch_size_stats(result):
    db_stats = result.fetchone()
    return {
        'table_job_size': db_stats[0],
        'db_diskq_size': db_stats[1],
    }


def get_task_queue():
    for retry in range(20):
        taskmap = requests.get(
            "http://stable.disk-queller.disk.stable.qloud-d.yandex.net:30811/z/celery-tasks.json",
            headers={
                'user-agent': 'disk.monitors.diskq_job_counters (GSID: %s)' % getenv('GSID', '').replace('\n', ' '),
            }
        ).json()
        if taskmap:
            task_queue = dict()
            for match in taskmap['tasks']:
                task = match['id'].replace('.',  '-')
                queue = match['queue']
                task_queue[task] = queue
            return task_queue
    else:
        return None

def append_metric(name, value, all_metrics):
    all_metrics.append(("media.disk.qdb.counters.%s" % name, value))


def append_counter_to_output(name, counter, all_metrics):
    for task_name, task_count in counter.items():
        metric_name = "%s_%s" % (name, task_name) if name else task_name
        append_metric(metric_name, task_count, all_metrics)


def append_shard_to_output(shard_num, mpfs_queues, aggr_tasks, size_stats, all_metrics):
    prefix = "by_shard_%02d" % shard_num
    append_counter_to_output("%s_mpfs_aggr_queue" % prefix, mpfs_queues, all_metrics)
    append_counter_to_output("%s_aggr" % prefix, aggr_tasks, all_metrics)
    append_metric("%s_table_job_size" % prefix, size_stats['table_job_size'], all_metrics)
    append_metric("%s_db_diskq_size" % prefix, size_stats['db_diskq_size'], all_metrics)
    append_metric("%s_counters_change_queue_size" % prefix, size_stats['counters_change_queue_size'], all_metrics)


if __name__ == '__main__':
    pg_group = getenv('PG_GROUP', 'disk_diskq_db').strip('%')
    pg_user = getenv('PG_USER')
    pg_pass = getenv('PG_PASS')
    pg_db = getenv('PG_DB', 'diskqdb')
    pg_port = getenv('PG_PORT', '6432')
    client = PgClient(pg_group, pg_user, pg_pass, pg_db, pg_port)
    logging.getLogger().setLevel(logging.INFO)
    results_aggr_tasks = Counter()
    results_mpfs_queues = Counter()
    results_size_stats = Counter()
    task_queue = get_task_queue()
    all_metrics = []
    logging.info("Init completed")

    task_stats = client.execute(
        "SELECT * FROM job_counters",
        lambda r: r.fetchall(),
        dict,
        keep_shard=True, replica_only=True
    )
    logging.info("Task stats counted")
    size_stats = client.execute(
        ("SELECT pg_total_relation_size('public.job'), pg_database_size(%s)", (pg_db,)),
        fetch_size_stats,
        dict,
        keep_shard=True, replica_only=True
    )
    logging.info("Size stats counted")
    queue_size = client.execute(
        "SELECT count(*) FROM job_counters_queue",
        lambda r: r.fetchone()[0],
        dict,
        keep_shard=True, replica_only=True
    )
    logging.info("Queue size counted")
    for shard_num in size_stats:
        try:
            size_stats[shard_num]['counters_change_queue_size'] = queue_size[shard_num]
        except Exception as e:
            logging.error("queue_size exception: %s" % e)

    for shard_num in task_stats:
        aggr_tasks, mpfs_queues = collect_stats(task_stats[shard_num], shard_num, task_queue, all_metrics)
        append_shard_to_output(shard_num, mpfs_queues, aggr_tasks, size_stats[shard_num], all_metrics)
        results_aggr_tasks.update(aggr_tasks)
        results_mpfs_queues.update(mpfs_queues)
        results_size_stats.update(size_stats[shard_num])

    append_counter_to_output("mpfs_aggr_queue", results_mpfs_queues, all_metrics)
    append_counter_to_output("aggr", results_aggr_tasks, all_metrics)
    append_counter_to_output("", results_size_stats, all_metrics)
    logging.info("All metrics counted. Send to graphite...")
    graphite.send_batch(all_metrics)
