#!/usr/bin/python
# -*- coding: utf-8 -*-
import json
import logging
import requests
import traceback
from datetime import date, timedelta, datetime
import time
import math
import copy
from itertools import groupby
from os import getenv
from format_mail import format_mail
from send_mail import send_mail
from stat_widgets import is_stat_widget, get_stat_widget_data
from optparse import OptionParser


headers = {
    "Authorization": "OAuth " + getenv('ROBOT_TOKEN'),
    "Content-type": "application/json;charset=UTF-8"
}


def get_dashboard_config():
    # return json.loads(open("/Users/tolmalev/Downloads/dashboard_config", "r").read())
    r = requests.get('https://stat.yandex-team.ru/_v3/dashboard/DiskDashboard', headers=headers, verify=False)
    return json.loads(r.text)


def remove_start(str, prefix):
    if str.startswith(prefix):
        return str[len(prefix):]


def get_widget_data(href):
    req_href = href

    for prefix in ["https://charts.yandex-team.ru/preview/editor/",
                   "https://charts.yandex-team.ru/preview/wizard/",
                   "https://charts-qloud.yandex-team.ru/preview/wizard/",
                   "/preview/wizard/",
                   "/wizard/",
                   "/preview/editor/"]:
        if req_href.startswith(prefix):
            req_href = "/ChartEditor?name=" + remove_start(href, prefix)
            logger.info("Href changed " + href + " -> " + req_href)


    body = {
        "responseOptions": {
            "includeConfig": True,
            "includeDependentParams": True
        },
        "path": req_href
    }
    logging.info("Loading data for chart: " + req_href)
    logging.info("body for chart: " + json.dumps(body))
    r = requests.post('https://charts.yandex-team.ru/api/run',
                      data=json.dumps(body),
                      headers=headers,
                      verify=False)

    if r.status_code != 200:
        logger.error(u"bad status code for " + req_href + u" code=" + str(r.status_code) + ". response=" + r.text)
        raise Exception(u"bad status code for " + req_href + u" code=" + str(r.status_code))
    else:
        logging.info("received data for chart: " + req_href)

    final_href = href
    if "http" not in href:
        final_href = "https://charts.yandex-team.ru/ChartPreview" + href

    js = json.loads(r.text)
    if 'data' not in js or 'graphs' not in js['data'] or 'categories_ms' not in js['data']:
        raise Exception(u"bad data in response")

    return js['data']['graphs'], js['data']['categories_ms'], final_href


def process_all(dashboard, parse_history):
    errors = []

    for tab in dashboard['tabs']:
        tab_name = tab['title']
        for widget in tab['widgets']:
            widget_name = widget['data'][0]['title']
            errors += process_widget(tab_name, widget_name, widget, parse_history)

    return errors


def ms_to_day(ms):
    return ms / 1000 / 3600 / 24


def get_today_day():
    return ms_to_day(int(time.time() * 1000))


def is_week_slale(categories_ms):
    if len(categories_ms) < 2:
        return False

    last_day = ms_to_day(categories_ms[-1])
    prev_day = ms_to_day(categories_ms[-2])

    return (last_day - prev_day) == 7


def has_no_last_point(categories_ms):
    if len(categories_ms) == 0:
        return True

    last_day = ms_to_day(categories_ms[-1])
    today_day = get_today_day()

    ok_period = 1

    if is_week_slale(categories_ms):
        ok_period = 13

    return today_day - last_day > ok_period


def process_widget(tab_name, widget_name, widget, parse_history):
    strange_found = []

    try:
        if is_stat_widget(tab_name, widget_name):
            graphs, categories_ms, href = get_stat_widget_data(tab_name, widget_name)
        else:
            graphs, categories_ms, href = get_widget_data(widget['data'][0]['href'])

        logger.debug("OK: " + tab_name + "/" + widget_name)

        last_day = ms_to_day(categories_ms[-1])

        if get_today_day() - last_day > 30 * 3:
            #устаревший отчет
            logger.info("Skip widget because of old data: " + tab_name + "/" + widget_name)
            return []

        if has_no_last_point(categories_ms):
            strange_found += [{
                "type": "no_last_point",
                "tab_name": tab_name,
                "widget_name": widget_name,
                "last_point_date": datetime.fromtimestamp(categories_ms[-1] / 1000).strftime('%Y-%m-%d'),
                "href": href
            }]
            return strange_found

        if parse_history:
            idx_lst = range(0, len(categories_ms))
        else:
            idx_lst = [0]

        for i in idx_lst:
            if last_day - ms_to_day(categories_ms[-i]) > 3 * 35:
                break

            logger.info("Processing day: " + str(ms_to_day(categories_ms[-(i+1)])))

            if i == 0:
                categories_ms_part = categories_ms
                graphs_part = copy.deepcopy(graphs)
            else:
                categories_ms_part = categories_ms[0: -i]
                graphs_part = copy.deepcopy(graphs)
                for g in graphs_part:
                    g['data'] = g['data'][0:-i]

            strange_found += process_widget_impl(categories_ms_part, graphs_part, href, tab_name, widget_name)

        if len(strange_found) > 0:
            def get_date(x):
                return x['date_str']

            sorted_data = sorted(strange_found, key=get_date)
            groupped_by_date = groupby(sorted_data, key=get_date)

            result_logger.info("Found strange for widget " + tab_name + "/" + widget_name)
            result_logger.info("URL: " + href)
            result_logger.info("=====================================")
            for date, items in groupped_by_date:
                names = ""
                for item in items:
                    names += json.dumps(item)

                result_logger.info(date + ": " + names)
            result_logger.info("=====================================")

        return strange_found

    except Exception as e:
        logger.error(traceback.format_exc(e))
        logger.error("FAILED: " + tab_name + "/" + widget_name)
        return strange_found + [{
            "type": "failed_get_data",
            "tab_name": tab_name,
            "widget_name": widget_name,
            "message": repr(e)
        }]


def process_widget_impl(categories_ms, graphs, href, tab_name, widget_name):
    by_week_indexes = [len(categories_ms) - 1]

    for i in range(1, len(categories_ms) + 1):
        if ms_to_day(categories_ms[by_week_indexes[-1]]) - ms_to_day(categories_ms[-i]) >= 7:
            by_week_indexes += [len(categories_ms) - i]

        if ms_to_day(categories_ms[-1]) - ms_to_day(categories_ms[-i]) > 30 * 3:
            break

    by_week_indexes.reverse()
    strange_found = []

    for graph in graphs:
        if 'name' not in graph:
            name = graph['title']
        else:
            name = graph['name']

        full_name = tab_name + '/' + widget_name + "/" + name
        data = graph['data']

        categories_ms_for_graph = categories_ms
        for i in range(0, len(data)):
            if data[-(i + 1)] is None:
                categories_ms_for_graph = categories_ms[0:-(i + 1)]
            else:
                break

        if has_no_last_point(categories_ms_for_graph):
            if len(categories_ms_for_graph) == 0:
                last_point_date = "1970-00-00"
            else:
                last_point_date = datetime.fromtimestamp(categories_ms_for_graph[-1] / 1000).strftime('%Y-%m-%d')

            strange_found += [{
                "type": "no_last_point",
                "tab_name": tab_name,
                "widget_name": widget_name,
                "graph_name": name,
                "last_point_date": last_point_date,
                "date_str": datetime.fromtimestamp(time.time()).strftime('%Y-%m-%d'),
                "href": href
            }]
            continue

        deltas = []
        for i in range(1, len(by_week_indexes) - 2):
            prev_idx = by_week_indexes[i]
            new_idx = by_week_indexes[i + 1]

            if data[new_idx] is None or data[prev_idx] is None:
                continue

            delta = data[new_idx] - data[prev_idx]
            deltas += [delta]

        if len(deltas) <= 1:
            continue

        avg_delta = 1.0 * sum(deltas) / len(deltas)
        std_dev = math.sqrt(1.0 * sum([(d - avg_delta) ** 2 for d in deltas]) / len(deltas))
        logger.debug(full_name + ": " + str(avg_delta) + " " + str(std_dev))

        last_idx = by_week_indexes[-1]
        prev_idx = by_week_indexes[-2]

        ms = categories_ms[last_idx]
        date_str = datetime.fromtimestamp(ms / 1000).strftime('%Y-%m-%d')

        current_value = data[last_idx]
        prev_value = data[prev_idx]

        if current_value is None or prev_value is None:
            logger.info("Bad data in graph")
            continue

        delta = current_value - prev_value
        dev = math.sqrt((delta - avg_delta) ** 2)
        if dev > std_dev * 4 and max(prev_value, current_value) > 500:
            message = "Found big delta! " + date_str + " " + full_name + "    " \
                      + "current: " + str(current_value) \
                      + " prev: " + str(prev_value) \
                      + " delta: " + str(delta) \
                      + " avg_delta: " + str(avg_delta) \
                      + " std_dev: " + str(std_dev) \
                      + " href: " + href

            strange_found += [{
                "type": "strange_data",
                "date_str": date_str,
                "tab_name": tab_name,
                "widget_name": widget_name,
                "graph_name": name,
                "current_value": current_value,
                "prev_value": prev_value,
                "delta": delta,
                "avg_delta": avg_delta,
                "std_dev": std_dev,
                "href": href
            }]

            logger.info(message)
        else:
            message = "Not big delta! " + date_str + " " + full_name + "    " \
                      + "current: " + str(current_value) \
                      + " prev: " + str(prev_value) \
                      + " delta: " + str(delta) \
                      + " avg_delta: " + str(avg_delta) \
                      + " std_dev: " + str(std_dev) \
                      + " href: " + href
            logger.info(message)

    return strange_found


if __name__ == '__main__':
    logging.basicConfig(
        level=logging.INFO,
        format="tskv\ttskv_format=ydisk-sandbox-default-log\tname=%(name)s\tunixtime=%(created)d\t"
               "timestamp=%(asctime)s\ttimezone=+0300\tlevel=%(levelname)s\tmessage=%(message)s")
    logger = logging.getLogger('disk-product-monitor')
    result_logger = logging.getLogger('disk-product-monitor-result')

    parser = OptionParser(usage="usage: ", version="%prog 1.0")
    parser.add_option("-b", "--back_history",
                      action="store_true",
                      dest="back_history",
                      default=False,
                      help="Check old data")

    (options, args) = parser.parse_args()

    errors = process_all(get_dashboard_config(), options.back_history)

    with open("erros.json", "wt") as f:
        f.write(json.dumps(errors))
        f.flush()

    send_mail(format_mail([e for e in errors if e["tab_name"] == u"Аудитория"]).encode("UTF-8"), subject_prefix="Аудитория")
    send_mail(format_mail(errors, template="mail_full.jinja2").encode("UTF-8"), to=["tolmalev@yandex-team.ru", "zanzara@yandex-team.ru", "aslonovskiy@yandex-team.ru"])

    # send_mail(format_mail([e for e in errors if e["tab_name"] == u"Аудитория"]).encode("UTF-8"), subject_prefix="Аудитория", to=["tolmalev@yandex-team.ru"])
