#!/usr/bin/python
# -*- coding: utf-8 -*-

from datetime import date, timedelta, datetime
from os import getenv
import requests
import time
import json
from itertools import groupby

headers = {
    "Authorization": "OAuth " + getenv('ROBOT_TOKEN'),
    "Content-type": "application/json;charset=UTF-8"
}


def get_stat_dictionary(dict_name):
    url = "https://stat.yandex-team.ru/_api/dictionary?_format=json&language=&name=vcfs%3A%3A" + dict_name
    r = requests.get(url, headers=headers, verify=False)

    if r.status_code != 200:
        raise Exception(u"bad status code for dict" + dict_name)

    return json.loads(r.text)


def build_create_sources_config():
    groups = get_stat_dictionary("disk_create_sources_groups")
    data = []

    for name, lst in groupby(sorted(groups.items(), key=lambda x: x[1]), lambda x: x[1]):
        path = ["\\t_total_\\t" + v[0] + "\\t" for v in lst]
        if len(path) == 1:
            path = path[0]

        data += [
            {
                "name": name,
                "filters": {
                    "path": path
                }
            },
        ]

    return {
        "tab_name": u"Аудитория",
        "widget_name": u"Новые пользователи",
        "href": "https://charts.yandex-team.ru/preview/editor/DiskDashboard/CreateSources",
        "sources": [
            {
                "path": "/Disk/Adhoc/DiskCreateSources",
                "scale": "d",
                "value_field": "users",
                "common_filters": {
                    "karma": "_good_"
                },
                "data": data
            }
        ]
    }


stat_widgets_config = [
    {
        "tab_name": u"Аудитория",
        "widget_name": u"Аудитория Диска (только залогин)",
        "href": "https://charts.yandex-team.ru/preview/editor/DiskDashboard/Audience_v2",
        "sources": [
            {
                "path": "/Disk/Special/Audience_v2",
                "scale": "d",
                "value_field": "users",
                "common_filters": {
                    "puid_type": "_total_",
                    "is_payment": "_total_"
                },
                "data": [
                    {
                        "name": u"Активные (всего)",
                        "filters": {
                            "platform": "_total_",
                            "action": "\\t_total_\\tactive\\t"
                        }
                    },
                    {
                        "name": u"Активные в личных интерфейсах",
                        "filters": {
                            "platform": "_total_",
                            "action": "\\t_total_\\tactive\\tЛичные интерфейсы\\t"
                        }
                    },
                    {
                        "name": u"Веб",
                        "filters": {
                            "platform": "web",
                            "action": "\\t_total_\\tactive\\t"
                        }
                    },
                    {
                        "name": u"Публичная страница",
                        "filters": {
                            "platform": "_total_",
                            "action": "\\t_total_\\tactive\\tПубличная страница\\t"
                        }
                    },
                    {
                        "name": u"Веб (личный)",
                        "filters": {
                            "platform": "web",
                            "action": "\\t_total_\\tactive\\tЛичные интерфейсы\\t"
                        }
                    },
                    {
                        "name": u"Десктопное ПО",
                        "filters": {
                            "platform": "desktop",
                            "action": "\\t_total_\\tactive\\t"
                        }
                    },
                    {
                        "name": u"Мобильные приложения",
                        "filters": {
                            "platform": "mobile",
                            "action": "\\t_total_\\tactive\\t"
                        }
                    },
                    {
                        "name": u"Просмотрщик документов",
                        "filters": {
                            "platform": "_total_",
                            "action": "\\t_total_\\tactive\\tПросмотрщик документов\\t"
                        }
                    },
                    {
                        "name": u"Android",
                        "filters": {
                            "platform": "android",
                            "action": "\\t_total_\\tactive\\t"
                        }
                    },
                    {
                        "name": u"iOS",
                        "filters": {
                            "platform": "iOS",
                            "action": "\\t_total_\\tactive\\t"
                        }
                    },
                    {
                        "name": u"Сохранение из Яндекс.Картинок",
                        "filters": {
                            "platform": "_total_",
                            "action": "\\t_total_\\tactive\\timage_save_to_disk\\t"
                        }
                    },
                    {
                        "name": u"Приложение файла к письму в Почте",
                        "filters": {
                            "platform": "_total_",
                            "action": "\\t_total_\\tactive\\tПриложение файла к письму в Почте\\t"
                        }
                    },
                    {
                        "name": u"Сохранение вложения в Диск",
                        "filters": {
                            "platform": "_total_",
                            "action": "\\t_total_\\tactive\\tСохранение вложения в Диск\\t"
                        }
                    },
                    {
                        "name": u"Элементы",
                        "filters": {
                            "platform": "_total_",
                            "action": "\\t_total_\\tactive\\tЭлементы\\t"
                        }
                    },
                ]
            }
        ]
    },
    build_create_sources_config()
]


def find_stat_widget_config(tab_name, widget_name):
    for c in stat_widgets_config:
        if c["tab_name"] == tab_name and c["widget_name"] == widget_name:
            return c

    return None


def is_stat_widget(tab_name, widget_name):
    if find_stat_widget_config(tab_name, widget_name):
        return True
    return False


def to_utf(s):
    if isinstance(s, str):
        return unicode(s, "utf-8")
    else:
        return s


def str_eq(a, b):
    if a is None and b is None:
        return True
    if a is None or b is None:
        return False

    return to_utf(a) == to_utf(b)


def matches_filter(item, filter):
    for k, v in filter.iteritems():
        if isinstance(v, list):
            found = False
            for sub_v in v:
                if str_eq(item.get(k, None), sub_v):
                    found = True
            if not found:
                return False
        else:
            if not str_eq(item.get(k, None), v):
                return False

    return True


def get_stat_widget_data(tab_name, widget_name):
    conf = find_stat_widget_config(tab_name, widget_name)
    if not conf:
        raise ValueError("Unknown stat widget " + tab_name + " / " + widget_name)

    date_min = datetime.fromtimestamp(time.time() - 86400 * 30 * 3).strftime('%Y-%m-%d')
    date_max = datetime.fromtimestamp(time.time()).strftime('%Y-%m-%d')

    data_by_graph = {}

    for source in conf["sources"]:
        url = "https://stat.yandex-team.ru/_api/statreport/tskv" \
              + source["path"] \
              + "?scale=" + source["scale"] \
              + "&date_min=" + date_min \
              + "&date_max=" + date_max

        r = requests.get(url, headers=headers, verify=False)
        if r.status_code != 200:
            raise Exception(u"bad status code for " + tab_name + u" / " + widget_name + u" code=" + str(r.status_code))

        line_n = 0
        for line in r.text.split("\n"):
            if line.strip() == "":
                continue

            line_n +=1
            value = parse_tskv(line)

            if matches_filter(value, source["common_filters"]):
                for graph in source["data"]:
                    if matches_filter(value, graph["filters"]):
                        ms = int(value["fielddate__ms"]) + 4 * 3600 * 1000
                        name = graph["name"]
                        data_value = float(value[source["value_field"]])

                        if name not in data_by_graph:
                            data_by_graph[name] = {}

                        if ms not in data_by_graph[name]:
                            data_by_graph[name][ms] = 0

                        data_by_graph[name][ms] += data_value

    all_categories_ms = set()
    for graph_name, data_by_date in data_by_graph.iteritems():
        all_categories_ms.update([int(k) for k in data_by_date.keys()])

    from_ms = min(all_categories_ms)
    to_ms = max(all_categories_ms)

    categories_ms = [from_ms + i * 86400 * 1000 for i in range(1 + (to_ms - from_ms) / 86400 / 1000)]
    href = conf["href"]

    graphs = []
    for source in conf["sources"]:
        for graph_config in source["data"]:
            data_by_date = data_by_graph.get(graph_config["name"], {})
            graph = {
                "name": graph_config["name"],
                "data": []
            }
            for ms in categories_ms:
                graph["data"].append(data_by_date.get(ms, None))

            graphs.append(graph)

    return graphs, categories_ms, href


def parse_tskv(line):
    parts = line.split("\t")
    value = {}
    for part in parts:
        eq_idx = part.find('=')
        k = part[0:eq_idx]
        v = part[(eq_idx + 1):]
        value[k] = v
    return value

# print get_stat_widget_data(u"Аудитория", u"Новые пользователи")
