#!/usr/bin/env python2.7
# -*- encoding: utf-8 -*-

import argparse
import re
import requests
import sqlite3
import sys
import time

from collections import defaultdict
from itertools import chain
from os import getenv
from os import path as os_path

dir_path = '/'.join(os_path.abspath(__file__).split('/')[:-2])
sys.path = [dir_path] + sys.path

from common import mailer
from yasmapi import GolovanRequest  # https://a.yandex-team.ru/arc/trunk/arcadia/infra/yasm/yasmapi

YASM_REQ = 'http://yasm.yandex-team.ru/metainfo/tags/prj?itype=qloud'
BASE_PATTERN = re.compile('^(disk\.|development\.disk)(?!.*?disk-front.pr-\d+)')

PRJ_PATTERNS = (('prod', re.compile('^disk\..*((?<!pre)stable|production)')),
                ('test', re.compile('^disk\..*(testing|prestable)')),
                ('misc', re.compile('.*')),)

SIG_PERIOD = 300
DAY = 86400
HOST = "QLOUD"
ITYPE = 'qloud'

# Собираемые метрики
SIGNALS = ('portoinst-cpu_limit_cores_tmmv',
           'portoinst-memory_limit_gb_tmmv',
           'portoinst-io_limit_bytes_tmmv',
           'portoinst-net_limit_mb_summ',)

TIME_FMT = '%Y-%m-%d %H:%M:%S'
# Глобальное время изменения, да, это нехорошо
TIMES = (None, None)


class Signal:
    allowed_params = {'itype', 'hosts', 'prj', 'ctype', 'geo', 'tier', 'graphs', 'charts', 'signals'}
    uniq_params = {'graphs', 'charts', 'signals'}

    @classmethod
    def generate(cls, **kwargs):
        for k in kwargs:
            if k not in cls.allowed_params:
                raise AttributeError('{} is not in allowed parameters: {}'.format(k, cls.allowed_params))

        sig_key = set(kwargs.keys()).intersection(cls.uniq_params)
        if len(sig_key) != 1:
            raise AttributeError('Set only one of allowed signal parameters: {}'.format(cls.uniq_params))

        sig = kwargs.pop(sig_key.pop())

        sig_string = '{}:{}'.format(';'.join('{}={}'.format(k, v) for k, v in kwargs.items()), sig)
        return sig_string


# SQL калькулятор статистики
class Analyzer:
    def __init__(self, columns, tablename='T'):
        """

        :param columns: tuple of (column, type)
        :param data:
        """

        self.columns = ()
        self.init_db(columns, tablename)

    def init_db(self, columns, tablename):
        self.db = sqlite3.connect(':memory:')
        self.tablename = tablename
        cursor = self.db.cursor()
        cursor.execute('create table {} ({});'.format(
            self.tablename,
            ', '.join(['{} {}'.format(f_name, f_type) for f_name, f_type in columns])
        ))
        self.columns = [f_name for f_name, _ in columns]

    def insert(self, data, single=False):
        cursor = self.db.cursor()
        try:
            if single:
                cursor.execute(
                    'insert into {} values ({})'.format(
                        self.tablename,
                        ','.join(['?' for _ in self.columns])),
                    data
                )
            else:
                cursor.executemany(
                    'insert into {} values ({})'.format(
                        self.tablename,
                        ','.join(['?' for _ in self.columns])),
                    data
                )
        except Exception as e:
            raise e
        self.db.commit()

    def select(self, request):
        request = re.sub(r'^\s*select\s*', '', request)
        request = 'select {}'.format(request)
        cursor = self.db.cursor()
        cursor.execute(request)
        return cursor.fetchall()

    def get_all_data(self, ):
        return self.select('* from T')


# Форматирование численных значений в клетках
def fmt_val(val, sign=False, rough=False):
    sign = '+' if sign else ''
    if isinstance(val, float):
        if int(val) == val:
            val = int(val)

    return '{val:{fmt}}'.format(val=val,
                                fmt=
                                '%s,' % sign if isinstance(val, int) else
                                '%s,.2f' % sign if isinstance(val, float) else
                                ''
                                )


# Получение всех prj
def get_all_projects(limit=-1):
    r = requests.get(YASM_REQ, headers={
            'user-agent': 'disk.monitors.qloud_resources (GSID: %s)' % getenv('GSID', '').replace('\n', ' '),
        })
    raw_projects = r.json()['response']['result']
    raw_projects = filter(lambda x: BASE_PATTERN.search(x), raw_projects)
    projects = defaultdict(list)
    for prj in raw_projects:
        for key, pattern in PRJ_PATTERNS:
            if pattern.search(prj):
                projects[key].append(prj)
                limit -= 1
                if not limit:
                    return dict(projects)
                break

    return dict(projects)


# Сбор сигналов в интервале, 1 prj, несколько signals
def prj_sig_changes(prj, signals, start_time, end_time, period):
    global TIMES

    signals_list = [Signal.generate(itype=ITYPE, prj=prj, signals=sig) for sig in signals]

    max_tries = 3
    for n_try in range(max_tries):
        try:
            old_values = list(GolovanRequest(HOST, period, start_time - period, start_time, signals_list))
            new_values = list(GolovanRequest(HOST, period, end_time - period, end_time, signals_list))
            break
        except Exception as e:
            if n_try < max_tries - 1:
                time.sleep(5)
                continue
            raise e

    # Костыль на глобальное время сигналов
    TIMES = tuple(map(lambda x: time.strftime(TIME_FMT, time.localtime(x[0][0])),
                      (old_values, new_values)))

    changes = []
    for sig, key in zip(signals, signals_list):
        old_sig_val, new_sig_val = map(lambda x: x[0][1][key],
                                       (old_values, new_values))
        changes.append((prj, sig, old_sig_val, new_sig_val))

    return changes


# Сбор сигналов по всем prj
def get_changes(projects, start_time=None, period=SIG_PERIOD, interval=DAY):
    if not start_time:
        if period < 3600:  # Пятиминутные данные, расчитываются раз в 25 минут, нужно брать чуть позже
            et = int(time.time() - 2 * period)
        else:
            et = int(time.time())
        st = et - interval
    else:
        st = start_time
        et = start_time + interval

    changes = []
    for kind, prjs in projects.items():
        for prj in prjs:
            for sc in prj_sig_changes(prj, SIGNALS, st, et, period):
                changes.append(tuple(chain(
                    (kind,),
                    sc
                )))
            time.sleep(5)

    return changes


def render_table(changes, headers, value_colons, color_offset=None, percent_columns=()):
    """

    :param changes:
    :param headers:
    :param color_offset: accepts color for background in format #1199ff or None
    :param percent_columns: unfair columns
    :return:
    """

    table = []
    table.extend([
        u"<table cellpadding=2 style='border-collapse: collapse; word-break: keep-all;'>\n",
        u"<tr style='border-top: 0; border-bottom: 2px solid black;'>"
    ])
    for header in headers:
        table.append(u"<th>%s</th>" % header)
    table.append(u"</tr>\n")

    last_row = ()
    for row in changes:
        table.append("<tr>")

        i = 0
        for i, (column, old_row) in enumerate(zip(row, last_row)):
            if last_row and column == old_row:
                # Форматирование цветовых отступов для повторяющихся полей
                if color_offset:
                    table.append('<td><font {clr}>{fmt}</font></td>'.format(fmt=fmt_val(column, rough=True),
                                                                            clr='color="{}"'.format(color_offset)))
                else:
                    table.append('<td>{fmt}</td>'.format(fmt=fmt_val(column, rough=True)))
            else:
                break

        for named_col in row[i:value_colons]:
            table.append('<td {align}>{fmt}</td>'.format(fmt=fmt_val(named_col, rough=True),
                                                         align="align='right'" if isinstance(named_col,
                                                                                             (int, float)) else ''))

        for c_num, value_col in enumerate(row[value_colons:]):
            table.append("<td {fmt} align='right'>{val}{perc}</td>".format(
                val=fmt_val(value_col, True),
                fmt="bgcolor='{}'".format(
                    '#ffd2dc' if value_col > 0 else '#ddffe8'
                ) if value_col else '',
                perc='%' if c_num in percent_columns else '',
            ))

        table.append("</tr>")
        last_row = row[:value_colons]
    table.append("</table>")

    return u'\n'.join(table)


if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    parser.add_argument('-a', '--address', default=getenv("RECIPIENT"), type=str, help="Mail address to send report")
    parser.add_argument('-m', '--metrics', default=SIGNALS, type=tuple, help="List of metrics")
    parser.add_argument('-t', '--time', default=DAY, help="Time interval in seconds")
    parser.add_argument('-d', '--diff', default=0.1, type=float, help="Minimum signals difference")
    parser.add_argument('-p', '--perc', default=1, type=float, help="Minimum signals percent difference")
    parser.add_argument('-s', '--signal', default=300, type=int, help="Metric period",
                        choices=(5, 300, 3600, 10800, 21600, 43200, 86400))
    parser.add_argument('-l', '--limit', default=-1, help="Limit maximum changes amount [debug purposes]", type=int)

    args = vars(parser.parse_args())
    if not args['address']:
        raise Exception("Email address is not set.")
    if isinstance(args['metrics'], basestring):
        args['metrics'] = args['metrics'].split(',')

    anlz = Analyzer((
        ('kind', 'text'),
        ('prj', 'text'),
        ('signal', 'text'),
        ('old_value', 'float'),
        ('new_value', 'float'),
    ))
    anlz.insert(get_changes(get_all_projects(limit=args['limit']), interval=args['time']))

    data = anlz.select('''
        select
            kind,
            signal,
            prj,
            old_value,
            new_value,
            new_value - old_value as diff,
            case
                when old_value = 0 and new_value = 0 then 0
                when old_value != 0 then 100 * ((new_value / old_value) - 1)
                else 100
            end as perc
        from T
        where
            abs(diff) >= {diff} and
            abs(perc) >= {perc}
        order by
            case
                when kind = 'prod' then 0
                when kind = 'test' then 1
                when kind = 'misc' then 2
                else kind
            end,
            signal,
            perc desc
        '''.format(diff=args['diff'],
                   perc=args['perc']))

    if data:
        tbl = render_table(data, ['kind', 'metric', 'prj', TIMES[0], TIMES[1], 'diff', 'diff %'], -2, '#d8d8d8', (1,))

        header = 'Qloud resources changes from {}'.format(*TIMES)
        mailer.send_email(
            args['address'],
            "Qloud resources changes",
            '<h3>{}</h3>\n'.format(header) + tbl
        )
