#!/usr/bin/env python2.7
# -*- coding: utf-8 -*-
import re
import sys
from collections import defaultdict
from os import getenv
from os import path as os_path

from nile.api.v1 import clusters
from nile.api.v1.datetime import Datetime, next_day
from qb2.api.v1 import extractors, filters

dir_path = '/'.join(os_path.abspath(__file__).split('/')[:-2])
sys.path = [dir_path] + sys.path
from common.startrek import ErrorTicket  # noqa
from common.staff import YandexStaffClient  # noqa


class SSHLogins(object):
    table_name = 'qloud_ssh_logins'
    qloud_logs = '//home/logfeller/logs/qloud-runtime-log/1d/'

    def __init__(self, yql_token, job_root, project='disk'):
        self.cluster = clusters.Hahn(yql_token=yql_token)
        self.result_table = job_root + '/' + self.table_name
        self.project = project
        self.yesterday = next_day(Datetime.now(), offset=-1)
        self.result_stream = None

    def generate(self):
        message_start = "Accepted publickey for "
        re_message = re.compile(
            message_start + "(?P<username>\\S+) from .*? ssh2: (?P<fingerprint>.*?) \\(qloud-env.*"
        )
        job = self.cluster.job()
        src_log = job.table(self.qloud_logs + self.yesterday)
        records = src_log.qb2(
            log='generic-yson-log',
            fields=[
                extractors.log_fields(
                    'iso_eventtime',
                    'qloud_project',
                    'qloud_application',
                    'qloud_environment',
                    'qloud_instance',
                    'host'
                ),
                extractors.log_field('message').hide(),
                extractors.log_field('threadName').hide(),
                extractors.custom(
                    'username', lambda m: re_message.search(m).group('username'), 'message').add_hints(type=str),
                extractors.custom(
                    'fingerprint',
                    lambda m: re_message.search(m).group('fingerprint'),
                    'message'
                ).add_hints(type=str),
            ],
            filters=[
                filters.equals('qloud_project', self.project),
                filters.or_(
                    filters.equals('qloud_environment', 'production'),
                    filters.equals('qloud_environment', 'corp-stable'),
                    filters.startswith('qloud_environment', 'stable'),
                ),
                filters.equals('threadName', 'LOG_AUTH'),
                filters.contains('message', message_start),
            ],
        ).sort(
            'iso_eventtime',
            'qloud_project',
            'qloud_application',
            'qloud_environment',
            'qloud_instance',
            'username',
        )
        self.result_stream = records.put(self.result_table, append=True)
        job.run()

    def read(self):
        if self.result_stream is None:
            return []
        return filter(lambda r: r['iso_eventtime'] >= self.yesterday, self.result_stream.read())

    def filter_legit_users(self, disk_users):
        legit_users = disk_users | {'root'}
        return filter(lambda r: r['username'] not in legit_users, self.read())


def group_by_app(logins_table):
    result = defaultdict(list)
    for row in logins_table:
        result[row['qloud_application']].append(row)
    return result


def json_table_to_tskv(json_table):
    for row in json_table:
        yield "\t".join('%s=%s' % i for i in row.to_dict().items())


if __name__ == '__main__':
    ssh_logins = SSHLogins(getenv("DISK_STAT_YQLTOKEN"), getenv("JOB_ROOT"), 'disk')
    ssh_logins.generate()
    staff_client = YandexStaffClient(getenv("STAFF_TOKEN"))
    disk_users = staff_client.get_users_by_groups(getenv("DISK_GROUPS").split(','))
    if getenv("DISK_USERS"):
        disk_users |= set(getenv("DISK_USERS").split(','))

    ticket_followers = set(staff_client.get_group_members('yandex_mnt_person_mailservice_service'))
    non_legit_logins = ssh_logins.filter_legit_users(disk_users)
    for app, logins in group_by_app(non_legit_logins).items():
        followers = ticket_followers | set(l['username'] for l in logins)
        ticket = ErrorTicket(u'Посторонний логин в qloud %s' % app, followers=followers)
        description = u'%%%%\n%s\n%%%%' % "\n".join(json_table_to_tskv(logins))
        ticket.create(description=description, components=[u"Администрирование"])
