# -*- coding: utf-8 -*-
import logging
import os
import re
import sys

import numpy as np

parent_dir_path = '/'.join(os.path.abspath(__file__).split('/')[:-2])
sys.path = [parent_dir_path] + sys.path

from common import graphite
from streaming.smotritel import http_get, HttpRequestException, test_playback, TimingPlaybackHandler, \
    LoggingPlaybackHandler

MAX_SEGMENT_COUNT = 20
GROUP_SIZE = 10
GLOBAL_GRAPHITE_PREFIX = 'media.disk.streaming.smotritel.etalon'


class Values:
    def __init__(self):
        self._values = []

    def append(self, value):
        self._values.append(value)

    @property
    def sum(self):
        return sum(self._values)

    @property
    def avg(self):
        return self.sum / len(self) if len(self) > 0 else 0

    def __len__(self):
        return len(self._values)

    def percentiles(self, *percentages):
        if len(self) > 0:
            a = np.array(self._values)
            return {p: np.percentile(a, p) for p in percentages}
        else:
            return {p: 0 for p in percentages}

    def send_percentiles(self, graphite_prefix, *percentages):
        percentiles = self.percentiles(*percentages)
        graphite.send_batch([
            (GLOBAL_GRAPHITE_PREFIX + '.' + graphite_prefix + '.p' + str(p), value)
            for p, value in percentiles.iteritems()
        ])

    def send_avg(self, graphite_prefix):
        self._send_one(graphite_prefix, self.avg)

    def send_sum(self, graphite_prefix):
        self._send_one(graphite_prefix, self.sum)

    def send_len(self, graphite_prefix):
        self._send_one(graphite_prefix, len(self))

    def _send_one(self, key, value):
        graphite.send_one(GLOBAL_GRAPHITE_PREFIX + '.' + key, value)


class EtalonPlaybackHandler:
    def __init__(self):
        self._all_start_times = Values()
        self._all_segment_times = Values()
        self._current_segment_times = Values()
        self._freeze_times = Values()

    def on_start(self):
        self._current_segment_times = Values()

    def on_segment_success(self, **kwargs):
        if kwargs['index'] == 0:
            start_time = kwargs['timer'].duration('start', '0')
            self._all_start_times.append(start_time)
            self._send_metric('start_time.current', start_time)

        self._all_segment_times.append(kwargs['timer'].last_duration)
        self._current_segment_times.append(kwargs['timer'].last_duration)

        if kwargs['reserve_await_time'].current_await > 0:
            self._freeze_times.append(kwargs['reserve_await_time'].current_await)

        self._send_metric('segment_duration', kwargs['segment'].duration)
        self._send_metric('segment_download_time.current_avg', self._current_segment_times.avg)
        self._send_metric('freeze.current', kwargs['reserve_await_time'].current_await)

    # TODO: add кол-во зафризившихся роликов - сумма, среднее
    def send_aggregate_metrics(self):
        self._all_start_times.send_percentiles('start_time', *range(50, 101))
        self._all_start_times.send_avg('start_time.avg')
        self._all_segment_times.send_percentiles('segment_download_time', *range(50, 101))
        self._all_segment_times.send_avg('segment_time.avg')
        self._freeze_times.send_len('freeze.count')
        self._freeze_times.send_sum('freeze.sum')
        self._freeze_times.send_avg('freeze.avg')
        self._freeze_times.send_percentiles('freeze', *range(50, 101))

    def _send_metric(self, key, value):
        graphite.send_one(GLOBAL_GRAPHITE_PREFIX + '.' + key, value)


if __name__ == "__main__":
    logging.basicConfig(level=logging.INFO)

    def read_stids(path):
        with open(path) as f:
            content = f.readlines()
        return [x.strip() for x in content]

    all_stids = read_stids('streaming/stids.txt')
    grouped_stids = [all_stids[i:i + GROUP_SIZE] for i in xrange(0, len(all_stids), GROUP_SIZE)]
    for stids in grouped_stids:
        etalon_handler = EtalonPlaybackHandler()
        playback_handler = TimingPlaybackHandler(LoggingPlaybackHandler(), etalon_handler)
        for stid in stids:
            try:
                resp = http_get('https://streaming-int.qloud.disk.yandex.net/stidhelper/?stid=%s' % stid)
                match = re.search("<a target='_blank' href='([^']+?/playlist.m3u8)[^']*'>direct-with-cache", resp.text)

                if match is None:
                    logging.warning('Could not get playlist for stid = %s', stid)
                    continue

                playlist_url = match.group(1).replace('http://', 'https://') + '?delete_cache=true'
            except HttpRequestException:
                logging.error('Error while getting playlist URL')
                continue

            test_playback(playlist_url, playback_handler, max_segment_count=MAX_SEGMENT_COUNT)

        logging.info('Batch processed. Sending metrics')
        etalon_handler.send_aggregate_metrics()
