#!/usr/bin/python

import logging
import re
import warnings
from optparse import OptionParser, Option

import requests
import sys
import time


class HttpRequestException(Exception):
    def __init__(self, status_code=None, message='', cause=None):
        if status_code is not None:
            message += 'Expected status code = 200, got = ' + str(status_code)

        if cause is not None:
            message += ', caused by ' + repr(cause)

        super(HttpRequestException, self).__init__(message)
        self.status_code = status_code


def http_get(url, timeout=30):
    try:
        level = logging.getLogger('requests').level
        logging.getLogger('requests').setLevel(logging.ERROR)
        try:
            resp = requests.get(url, timeout=timeout, verify=False)
        finally:
            logging.getLogger('requests').setLevel(level)

        if resp.status_code != 200:
            logging.warn('Failed to download %s: response code = %d', url, resp.status_code)
            raise HttpRequestException(status_code=resp.status_code)

        return resp
    except requests.exceptions.RequestException as e:
        logging.info('Failed to download %s: %s', url, e)
        raise HttpRequestException(cause=e), None, sys.exc_info()[2]


class Segment:
    def __init__(self, url, duration, offset_time):
        self.url = url
        self.duration = duration
        self.offset_time = offset_time

    def fetch(self):
        http_get(self.url)


class Playlist:
    def __init__(self, segments):
        self.segments = segments

    @staticmethod
    def fetch(url):
        resp = http_get(url)

        base_url = url[0:url.rfind('/') + 1]

        def resolve_url(unresolved_url):
            return unresolved_url if re.match('^(http|https)://', unresolved_url) else base_url + unresolved_url

        segment_data = re.findall('#EXTINF:(\d+),\s*\n\s*(\S+)\s*', resp.text, re.MULTILINE)
        offset_time = 0
        segments = []
        for (seg_duration, seg_url) in segment_data:
            segment = Segment(resolve_url(seg_url), int(seg_duration), offset_time)
            offset_time += segment.duration
            segments.append(segment)

        return Playlist(segments)


class Timer:
    def __init__(self):
        self._times = {}
        self._next_to_last = None
        self._last = None

    def rec(self, key):
        self._times[key] = time.time()
        self._next_to_last = self._last
        self._last = self._times[key]

    def duration(self, start_key, end_key):
        return self._times[end_key] - self._times[start_key]

    @property
    def last_duration(self):
        return self._last - self._next_to_last


class ReserveAwaitTime:
    def __init__(self):
        self.reserve = 0
        self.current_await = 0
        self.total_await = 0

    def add(self, segment_duration, fetch_duration, index):
        if index == 0:
            self.reserve = segment_duration
            return

        delta = segment_duration - fetch_duration
        self.reserve += delta
        if self.reserve >= 0:
            self.current_await = 0
        else:
            self.current_await = abs(self.reserve)
            self.reserve = 0
        self.total_await += self.current_await

    @property
    def reserve_or_negative_await(self):
        return -self.current_await if self.current_await else self.reserve


class TimingPlaybackHandler:
    def __init__(self, *handlers):
        self._handler = HandlerWrapper(*handlers)
        self._timer = Timer()
        self._reserve_await_time = ReserveAwaitTime()

    def on_start(self):
        self._timer.rec('start')
        self._handler.on_start()

    def on_playlist_error(self, **kwargs):
        self._handler.on_playlist_error(kwargs)

    def on_playlist_success(self, **kwargs):
        self._timer.rec('playlist')
        self._handler.on_playlist_success(timer=self._timer, **kwargs)

    def on_segment_error(self, **kwargs):
        self._handler.on_segment_error(kwargs)

    def on_segment_success(self, **kwargs):
        segment = kwargs['segment']
        index = kwargs['index']

        self._timer.rec(str(index))
        self._reserve_await_time.add(segment.duration, self._timer.last_duration, index)

        self._handler.on_segment_success(timer=self._timer, reserve_await_time=self._reserve_await_time, **kwargs)

    def on_end(self):
        self._handler.on_end()


class LoggingPlaybackHandler:
    def __init__(self):
        pass

    def on_playlist_error(self, **kwargs):
        logging.warning('Error while downloading playlist: %s', kwargs['error'])

    def on_playlist_success(self, **kwargs):
        logging.info('Downloaded playlist in %.2f', kwargs['timer'].duration('start', 'playlist'))

    def on_segment_success(self, **kwargs):
        index = kwargs['index']

        if index == 0:
            logging.info("Start delay (playlist download + 1st segment download): %.2f",
                         kwargs['timer'].duration('start', '0'))

        logging.info('Segment #%d downloaded in %.2f: reserve = %.2f, freeze = %.2f',
                     index,
                     kwargs['timer'].last_duration,
                     kwargs['reserve_await_time'].reserve,
                     kwargs['reserve_await_time'].current_await)


class HandlerWrapper:
    delegate_methods = ['on_start', 'on_end', 'on_playlist_error', 'on_playlist_success', 'on_segment_error', 'on_segment_success']

    def __init__(self, *handlers):
        self._handlers = handlers

    def __getattr__(self, item):
        if item in HandlerWrapper.delegate_methods:
            def delegator(*args, **kw):
                for handler in self._handlers:
                    getattr(handler, item, self._stub_method)(*args, **kw)
            return delegator
        else:
            raise AttributeError('No such method "' + item + "'")

    def _stub_method(self, *args, **kwargs):
        # do nothing
        pass


# log fail count
# segment wait count
# segment wait total time
def test_playback(playlist_url, handler, start_index=0, max_segment_count=None, playlist_retries=1, segment_retries=1):
    handler = HandlerWrapper(handler)
    handler.on_start()
    try:
        playlist = None
        for i in range(0, playlist_retries + 1):
            try:
                playlist = Playlist.fetch(playlist_url)
                handler.on_playlist_success(playlist=playlist)
                break
            except HttpRequestException as e:
                handler.on_playlist_error(error=e)

        if playlist is None:
            return

        if max_segment_count is None:
            max_segment_count = len(playlist.segments)

        segments = playlist.segments[start_index:(min(len(playlist.segments), max_segment_count - start_index) - 1)]
        segment_index = 0
        for segment in segments:
            for i in range(0, segment_retries + 1):
                try:
                    segment.fetch()
                    handler.on_segment_success(segment=segment, index=segment_index)
                    break
                except HttpRequestException as e:
                    handler.on_segment_error(segment=segment, index=segment_index, error=e)

            segment_index += 1
    finally:
        handler.on_end()


if __name__ == '__main__':
    logging.basicConfig(level=logging.INFO)

    opts_list = (
        Option(
            '-p', '--playlist',
            action='store',
            dest='playlist_url',
            type='string',
            default=None,
            help='Playlist URL'
        ),
        Option(
            '--segment-count',
            action='store',
            dest='segment_count',
            type='int',
            default=None,
            help='Max number of segments to download'
        ),
        Option(
            '--need-rewind',
            action='store_true',
            dest='rewind',
            default=True,
            help="run without commit",
        ),
    )

    opts_parser = OptionParser('Usage: ./smotritel.py -p {playlist_url}', option_list=opts_list)
    (opts, args) = opts_parser.parse_args()

    if not opts.playlist_url:
        opts_parser.print_usage()
        exit(1)

    logging.info('Processing playlist %s', opts.playlist_url)
    test_playback(opts.playlist_url, TimingPlaybackHandler(LoggingPlaybackHandler()),
                  start_index=0, max_segment_count=opts.segment_count)
