#!/usr/bin/env python2.7
# -*- coding: utf-8 -*-
import re
from datetime import datetime, timedelta
from os import getenv

from nile.api.v1 import clusters
from qb2.api.v1 import extractors, filters as qf

YT_LINK_PREFIX = u'https://yt.yandex-team.ru/hahn/navigation?path='
re_uuid = re.compile(r'\\"uuid\\":\\"([0-9a-zA-Z]+)\\"')
re_device_id = re.compile(r'"id":"([0-9a-zA-Z]+)"')


def fetch_uuid(user_agent):
    match = re_uuid.search(user_agent)
    if match:
        return match.group(1)
    return None


def fetch_device_id(headers):
    match = re_device_id.search(headers)
    if match:
        return match.group(1)
    return None


def generate_vertical_device_ids_table(yql_token, pwd, yesterday, uids):
    result_table = pwd + '/vertical_device_ids'

    sources = [
        result_table,
        '//home/logfeller/logs/ydisk-java-webdav-access-log/1d/%s' % yesterday,
        '//home/logfeller/logs/ydisk-mpfs-access-log/1d/%s' % yesterday,
    ]
    job = clusters.Hahn(yql_token=yql_token).env(parallel_operations_limit=3).job()
    records = []
    for table in sources:
        fields = [extractors.log_field('uid')]
        filters = [
            qf.one_of('uid', uids),
            qf.defined('DeviceID'),
            qf.not_(qf.equals('DeviceID', 'unknown')),
            qf.not_(qf.equals('DeviceID', 'ceaa1bbd7a2ab07f189da5145e9747c0')),  # MOBDISK-21389 Disable Huawei
            qf.not_(qf.equals('DeviceID', '9f1e2c8e61d4d5ff0dc15f7a640076a5')),  # MOBDISK-21389 Disable Huawei
            qf.not_(qf.equals('DeviceID', '5c614d04c9810679302792e82d2e62c9')),  # MOBDISK-21389 Disable Huawei
        ]
        if table == result_table:
            fields.append(extractors.log_field('DeviceID'))
        elif 'webdav' in table:
            fields.extend([
                extractors.log_field('user_agent').hide(),
                extractors.custom('DeviceID', fetch_uuid, 'user_agent').add_hints(type=str),
            ])
            filters.extend([
                qf.contains('user_agent', 'Yandex.Disk'),
            ])
        elif 'mpfs' in table:
            fields.extend([
                extractors.log_fields('appname', 'headers').hide(),
                extractors.custom('DeviceID', fetch_device_id, 'headers').add_hints(type=str),
            ])
            filters.extend([
                qf.equals('appname', 'platform'),
                qf.contains('headers', 'Yandex.Disk')
            ])

        src_log = job.table(table)
        records.append(src_log.qb2(log='generic-yson-log', fields=fields, filters=filters))

    job.concat(*records).unique('uid', 'DeviceID').sort('uid').put(result_table)
    job.run()
    return result_table


if __name__ == '__main__':
    yql_token = getenv("DISK_STAT_YQLTOKEN")
    pwd = getenv("JOB_ROOT")
    users = dict(i.split('=')[::-1] for i in getenv("UIDS").split(','))
    yesterday = getenv("DAY_FOR_COUNT") or str((datetime.now() - timedelta(1)).date())
    result_path = generate_vertical_device_ids_table(yql_token, pwd, yesterday, users.keys())
    print(YT_LINK_PREFIX + result_path)
